<?php

/**
 * Class ProdRules
 * Manage production rules and simple BOM generation
 */

require_once DOL_DOCUMENT_ROOT . '/core/class/commonobject.class.php';

class ProdRules extends CommonObject
{
    public $element = 'prod_rules';
    public $table_element = 'prod_rules';

    /** @var DoliDB */
    public $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Get list of rules with optional filter
     *
     * @param array $filter
     * @return array
     */
    public function fetchAll($filter = array())
    {
        global $user;

        $sql = "SELECT * FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " AND fk_user_creat = " . (int) $user->id;   // ⬅️ TYLKO swoje reguły

        if (!empty($filter['from_ref'])) {
            $sql .= " AND from_ref = '" . $this->db->escape($filter['from_ref']) . "'";
        }
        if (!empty($filter['to_ref'])) {
            $sql .= " AND to_ref = '" . $this->db->escape($filter['to_ref']) . "'";
        }
        if (!empty($filter['fk_ruleset'])) {
            $sql .= " AND fk_ruleset = " . (int) $filter['fk_ruleset'];
        }
        if (isset($filter['active'])) {
            $sql .= " AND active = " . (int) $filter['active'];
        }

        $sql .= " ORDER BY step_order, rowid";

        $res = $this->db->query($sql);
        if (!$res) {
            return array();
        }

        $out = array();
        while ($obj = $this->db->fetch_object($res)) {
            $out[] = $obj;
        }

        return $out;
    }
    /**
     * Insert one rule
     *
     * @param User  $user
     * @param array $data
     * @return int  new id or -1 on error
     */
public function createRule($user, $data)
{
    global $conf;

    $now = $this->db->idate(dol_now());

    $sql = "INSERT INTO " . MAIN_DB_PREFIX . "prod_rules(";
    $sql .= "from_ref, operation, to_ref, fk_ruleset, qty_per_from, workstation, step_order, ";
    $sql .= "is_final, is_raw, active, note, datec, fk_user_creat, entity";
    $sql .= ") VALUES (";

    // from_ref
    $sql .= "'" . $this->db->escape($data['from_ref']) . "',";

    // operation
    $sql .= "'" . $this->db->escape($data['operation']) . "',";

    // to_ref
    $sql .= "'" . $this->db->escape($data['to_ref']) . "',";

    // fk_ruleset
    if (!empty($data['fk_ruleset'])) {
        $sql .= (int) $data['fk_ruleset'] . ",";
    } else {
        $sql .= "NULL,";
    }

    // qty_per_from
    $qty = (isset($data['qty_per_from']) ? (float) $data['qty_per_from'] : 1.0);
    $sql .= $qty . ",";

    // workstation
    if (!empty($data['workstation'])) {
        $sql .= "'" . $this->db->escape($data['workstation']) . "',";
    } else {
        $sql .= "NULL,";
    }

    // step_order
    $sql .= (int) (isset($data['step_order']) ? $data['step_order'] : 10) . ",";

    // is_final
    $sql .= (int) (!empty($data['is_final']) ? 1 : 0) . ",";

    // is_raw
    $sql .= (int) (!empty($data['is_raw']) ? 1 : 0) . ",";

    // active
    $sql .= "1,";

    // note
    if (!empty($data['note'])) {
        $sql .= "'" . $this->db->escape($data['note']) . "',";
    } else {
        $sql .= "NULL,";
    }

    // datec
    $sql .= "'" . $this->db->escape($now) . "',";

    // fk_user_creat
    $sql .= (int) $user->id . ",";

    // entity
    $sql .= (int) $conf->entity;

    $sql .= ")";

    $res = $this->db->query($sql);
    if (!$res) {
        $this->error = $this->db->lasterror();
        return -1;
    }

    return $this->db->last_insert_id(MAIN_DB_PREFIX . "prod_rules");
}

    /**
     * Compute raw requirements for given target product and quantity
     * (simplified version)
     *
     * @param string $target_ref
     * @param float  $qty_target
     * @return array ref => qty
     */
    public function computeRequirements($target_ref, $qty_target)
    {
        global $user;

        $need  = array();
        $stack = array();

        $stack[] = array($target_ref, $qty_target);

        while (!empty($stack)) {
            list($ref, $qty) = array_pop($stack);

            // 1) Ustalamy, czy ref jest surowcem wg reguł TEGO użytkownika
            $sql = "SELECT is_raw FROM " . MAIN_DB_PREFIX . "prod_rules";
            $sql .= " WHERE from_ref = '" . $this->db->escape($ref) . "'";
            $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
            $sql .= " AND fk_user_creat = " . (int) $user->id;
            $sql .= " ORDER BY rowid DESC LIMIT 1";

            $res = $this->db->query($sql);
            $is_raw = 0;
            if ($res && ($obj = $this->db->fetch_object($res))) {
                $is_raw = (int) $obj->is_raw;
            }

            if ($is_raw) {
                if (!isset($need[$ref])) {
                    $need[$ref] = 0;
                }
                $need[$ref] += $qty;
                continue;
            }

            // 2) Szukamy reguł, które tworzą ten ref jako to_ref – też tylko dla tego usera
            $sql = "SELECT * FROM " . MAIN_DB_PREFIX . "prod_rules";
            $sql .= " WHERE to_ref = '" . $this->db->escape($ref) . "'";
            $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
            $sql .= " AND fk_user_creat = " . (int) $user->id;

            $res = $this->db->query($sql);
            if (!$res || $this->db->num_rows($res) == 0) {
                // Treat as raw if no rules for this user
                if (!isset($need[$ref])) {
                    $need[$ref] = 0;
                }
                $need[$ref] += $qty;
                continue;
            }

            while ($r = $this->db->fetch_object($res)) {
                $from_ref     = $r->from_ref;
                $qty_per_from = (float) $r->qty_per_from;
                if ($qty_per_from == 0) {
                    $qty_per_from = 1.0;
                }

                $needed_from = $qty / $qty_per_from;
                $stack[] = array($from_ref, $needed_from);
            }
        }

        return $need;
    }

    /**
     * Generate a simple BOM in Dolibarr for a given FG
     * (FG -> list of raw requirements)
     *
     * @param User   $user
     * @param string $fg_ref
     * @param float  $qty_fg
     * @return int   bom id or -1 on error
     */
    public function generateBomForProduct($user, $fg_ref, $qty_fg = 1.0, $job_number = null, $fk_ruleset = null, $raw_ref = null)
    {
        global $conf, $langs;
        // UWAGA: tutaj $user jest parametrem, ale zostawmy go – nie trzeba global

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        // 1) Pobierz produkt, dla którego robimy BOM
        $prod = new Product($this->db);
        $res = $prod->fetch('', $fg_ref);
        if ($res <= 0) {
            $this->error = 'Product with ref ' . $fg_ref . ' not found';
            return -1;
        }
    
        // 2) ZBUDUJ WYMAGANIA JEDNEGO POZIOMU:
        //    wszystkie reguły, gdzie to_ref = nasz produkt, TYLKO bieżący user
        $requirements = array();

        $sql  = "SELECT from_ref, qty_per_from, step_order";
        $sql .= " FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE to_ref = '" . $this->db->escape($fg_ref) . "'";
        $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " AND fk_user_creat = " . (int) $user->id;   // ⬅️ tu dopisujemy

        if (!empty($fk_ruleset)) {
            $sql .= " AND fk_ruleset = " . (int) $fk_ruleset;
        }

        $sql .= " ORDER BY step_order, rowid";

        $resql = $this->db->query($sql);
        if (! $resql) {
            $this->error = 'Error selecting rules: ' . $this->db->lasterror();
            return -1;
        }

        while ($obj = $this->db->fetch_object($resql)) {
            $ref = $obj->from_ref;

            if (!isset($requirements[$ref])) {
                $requirements[$ref] = array(
                    'qty'        => 0,
                    'step_order' => (int) $obj->step_order
                );
            }

            // Ilość komponentu = ilość FG w BOM * ilość z reguły
            $requirements[$ref]['qty'] += ((float) $qty_fg * (float) $obj->qty_per_from);

            // step_order – bierzemy najmniejszy krok, jeśli jest więcej niż jedna reguła
            if ((int) $obj->step_order < $requirements[$ref]['step_order']) {
                $requirements[$ref]['step_order'] = (int) $obj->step_order;
            }
        }

        
        // Dodaj RAW materiał jako komponent wejściowy (1:1 do ilości FG)
        if (!empty($raw_ref)) {
            if (!isset($requirements[$raw_ref])) {
                $requirements[$raw_ref] = array(
                    'qty'        => 0,
                    'step_order' => 0
                );
            }
            // RAW zużywany 1:1 względem FG
            $requirements[$raw_ref]['qty'] += (float) $qty_fg;

            // Upewnij się, że raw ma najniższy krok (na początku listy)
            if ($requirements[$raw_ref]['step_order'] > 0) {
                $requirements[$raw_ref]['step_order'] = 0;
            }
        }

// 3) Utwórz BOM (nagłówek)
        $bom = new Bom($this->db);

        if (!empty($job_number)) {
            $bom->ref   = 'BOM-' . $fg_ref . '-' . $job_number;
            $bom->label = 'Auto BOM for ' . $fg_ref . ' (Job ' . $job_number . ')';
        } else {
            $bom->ref   = 'BOM-' . $fg_ref;
            $bom->label = 'Auto BOM for ' . $fg_ref;
        }

        $bom->fk_product = $prod->id;
        $bom->qty        = $qty_fg;
        $bom->status     = 0;
        $bom->entity     = $conf->entity;

        $res = $bom->create($user);
        if ($res <= 0) {
            $this->error = $bom->error;
            return -1;
        }

        // 4) Dodaj linie BOM na podstawie $requirements
        foreach ($requirements as $ref => $info) {
            $lineqty    = $info['qty'];
            $step_order = $info['step_order'];

            // Znajdź produkt komponentu
            $p = new Product($this->db);
            if ($p->fetch('', $ref) <= 0) {
                // brak produktu – log i pomijamy
                dol_syslog("ProductionRules::generateBomForProduct missing product ".$ref, LOG_WARNING);
                continue;
            }

            $line = new BomLine($this->db);
            $line->fk_bom     = $bom->id;
            $line->fk_product = $p->id;
            $line->qty        = $lineqty;
            $line->position   = $step_order;

            $resline = $line->create($user);
            if ($resline <= 0) {
                $this->error = 'Error creating BOM line for '.$ref.' : '.$line->error;
                return -1;
            }
        }

        // 5) Zatwierdź BOM
        $bom->status = 1;
        $bom->update($user);

        return $bom->id;
    

    /**
     * Generate BOM for RuleSet SF using material pack numbers (1 sheet = 1 SF).
     * $raw_ref_list: comma separated numeric pack refs. Duplicates are NOT allowed.
     */
    public function generateBomForRuleset($user, $fk_ruleset, $sf_ref, $job_number = null, $raw_ref_list = null)
    {
        global $conf, $langs;

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        if (empty($fk_ruleset)) {
            $this->error = 'RuleSet is required';
            return -1;
        }
        if (empty($sf_ref)) {
            $this->error = 'SF ref is required';
            return -1;
        }
        if (empty($raw_ref_list)) {
            $this->error = 'Material ref is empty';
            return -1;
        }

        // Parse pack numbers: numeric only, unique
        $packs = array();
        $seen = array();
        $tokens = preg_split('/\s*,\s*/', trim($raw_ref_list));
        foreach ($tokens as $tok) {
            if ($tok === '') continue;

            if (!preg_match('/^\d+$/', $tok)) {
                $this->error = 'Material must be numeric pack numbers. Bad token: ' . $tok;
                return -1;
            }
            if (isset($seen[$tok])) {
                $this->error = 'Duplicate material pack number: ' . $tok;
                return -1;
            }
            $seen[$tok] = true;
            $packs[] = $tok;
        }

        if (empty($packs)) {
            $this->error = 'Material list is empty';
            return -1;
        }

        // SF product
        $prod = new Product($this->db);
        $res = $prod->fetch('', $sf_ref);
        if ($res <= 0) {
            $this->error = 'Product with ref ' . $sf_ref . ' not found';
            return -1;
        }

        $qty_sf = count($packs); // 1 sheet = 1 SF

        // Build routing lines
        $routingLines = array();
        $sql  = "SELECT DISTINCT step_order, operation";
        $sql .= " FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE fk_ruleset = " . ((int) $fk_ruleset);
        $sql .= " AND active = 1";
        $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " ORDER BY step_order ASC";

        $resql = $this->db->query($sql);
        if ($resql) {
            while ($obj = $this->db->fetch_object($resql)) {
                $step = (int) $obj->step_order;
                $op   = trim((string) $obj->operation);
                if ($op === '') continue;
                $routingLines[] = sprintf('%d %s', $step, $op);
            }
        }

        // Create BOM header
        $bom = new Bom($this->db);
        $bom->ref = !empty($job_number) ? ('BOM-' . $job_number) : ('BOM-' . $sf_ref . '-' . dol_print_date(dol_now(), '%Y%m%d-%H%M%S'));
        $bom->label = !empty($job_number) ? ('Auto BOM for ' . $sf_ref . ' (Job ' . $job_number . ')') : ('Auto BOM for ' . $sf_ref);
        $bom->quantity   = (float) $qty_sf;
        $bom->fk_product = $prod->id;
        $bom->fk_user    = $user->id;
        $bom->entity     = $conf->entity;

        // Preview / audit note
        $note = array();
        $note[] = 'Generated by ProductionRules';
        if (!empty($job_number)) $note[] = 'Job: ' . $job_number;
        $note[] = 'Qty SF: ' . $qty_sf;
        $note[] = '';
        $note[] = 'Material packs:';
        foreach ($packs as $p) $note[] = '- ' . $p;
        $note[] = '';
        $note[] = $langs->transnoentitiesnoconv('Routing') . ':';
        foreach ($routingLines as $l) $note[] = $l;

        $bom->note_public = implode("\n", $note);

        $resbom = $bom->create($user);
        if ($resbom <= 0) {
            $this->error = 'Error creating BOM: ' . $bom->error;
            return -1;
        }

        // Add one BOM line per pack (qty = 1)
        $pos = 10;
        foreach ($packs as $p) {
            $prodMat = new Product($this->db);
            $r = $prodMat->fetch('', $p);
            if ($r <= 0) {
                $this->error = 'Material product with ref ' . $p . ' not found';
                return -1;
            }

            $line = new BomLine($this->db);
            $line->fk_bom      = $bom->id;
            $line->fk_product  = $prodMat->id;
            $line->qty         = 1.0;
            $line->position    = $pos;
            $pos += 10;
            $line->fk_unit     = $prodMat->fk_unit;
            $line->desc        = $langs->transnoentitiesnoconv('Material pack for SF') . ' ' . $sf_ref;

            $resline = $line->create($user);
            if ($resline <= 0) {
                $this->error = 'Error creating BOM line for material ' . $p . ' : ' . $line->error;
                return -1;
            }
        }

        // Approve
        $bom->status = 1;
        $bom->update($user);

        return $bom->id;
    }

}
}
