<?php

/**
 * Render flow for one RuleSet.
 *
 * Oczekuje struktury z RuleSetFlowBuilder::build():
 *  - raw_ref      : string
 *  - chain        : array of rule objects (główna ścieżka, po step_order)
 *  - side_outputs : array from_ref => array(rule objects)  (boczne FG/SCRAP)
 */

class RuleSetFlowRenderer
{
    /**
     * @param array $flow Wynik z RuleSetFlowBuilder::build()
     */
    public static function render($flow)
    {
        global $langs;

        if (empty($flow) || empty($flow['raw_ref'])) {
            print '<br>';
            print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');
            print '<div class="pr-flow-container">';
            print '<span class="opacitymedium">No rules in this RuleSet.</span>';
            print '</div>';
            return;
        }

        $rawRef      = $flow['raw_ref'];
        $chain       = $flow['chain'];        // główna ścieżka (po step_order)
        $sideOutputs = $flow['side_outputs']; // boczne wyjścia

        // Zbiór wszystkich reguł (główna ścieżka + boczne) – przyda się
        // do określania typu node'a dla from_ref w bocznych gałęziach
        $allRules = $chain;
        foreach ($sideOutputs as $tmpFrom => $tmpRules) {
            foreach ($tmpRules as $tmpR) {
                $allRules[] = $tmpR;
            }
        }

        print '<br>';
        print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');

        // Tickbox do pokazywania komentarza (operation) obok ref
        print '<div class="pr-flow-options">';
        print '<label><input type="checkbox" id="pr-show-comments"> Show comment</label>';
        print '</div>';

        print '<div class="pr-flow-container">';

        // ----------------------------------------------------------------
        // 1) LINIA GŁÓWNA: RAW → SF (tylko pierwszy krok z chain)
        // ----------------------------------------------------------------
        print '<div class="pr-flow-row">';

        // RAW
        $tooltipRaw = self::buildRawTooltip($rawRef);
        print self::renderNode($rawRef, 'raw', null, $tooltipRaw);

        $firstRule = null;
        if (!empty($chain)) {
            $firstRule = $chain[0];

            // RAW → SF
            print '<span class="pr-arrow">→</span>';

            $typeFirst    = self::detectNodeType($firstRule);
            $tooltipFirst = self::buildRuleTooltip($firstRule, $typeFirst, $rawRef);

            print self::renderNode($firstRule->to_ref, $typeFirst, $firstRule, $tooltipFirst);
        }

        print '</div>'; // .pr-flow-row

        // ----------------------------------------------------------------
        // 2) Dalsza część głównego łańcucha (pozostałe kroki chain)
        // ----------------------------------------------------------------
        if (!empty($chain)) {
            print '<div class="pr-flow-row pr-flow-row-mainchain">';

            // Pusta komórka w miejscu RAW (żeby wyrównać)
            print '<span class="pr-node pr-node-empty">&nbsp;</span>';

            $prevTo = $firstRule->to_ref;

            foreach (array_slice($chain, 1) as $rule) {
                // jeżeli from_ref nie zgadza się z poprzednim to_ref,
                // to rysujemy tylko to_ref jako osobny node (bez strzałki)
                if ($rule->from_ref !== $prevTo) {
                    print '<span class="pr-arrow">&nbsp;</span>';
                    $type    = self::detectNodeType($rule);
                    $tooltip = self::buildRuleTooltip($rule, $type, $rawRef);
                    print self::renderNode($rule->to_ref, $type, $rule, $tooltip);
                } else {
                    // klasyczny przypadek: to_ref poprzedniej reguły
                    // jest from_ref kolejnej → rysujemy strzałkę i kolejny node
                    print '<span class="pr-arrow">→</span>';
                    $type    = self::detectNodeType($rule);
                    $tooltip = self::buildRuleTooltip($rule, $type, $rawRef);
                    print self::renderNode($rule->to_ref, $type, $rule, $tooltip);
                }

                $prevTo = $rule->to_ref;
            }

            print '</div>'; // .pr-flow-row-mainchain
        }

        // ----------------------------------------------------------------
        // 2b) Dodatkowe wyjścia spod SF w chain (np. alternatywne FG)
        // ----------------------------------------------------------------
        // Szukamy tych z chain, które są SF (from_ref != to_ref, is_final=0),
        // a mają boczne wyjścia w side_outputs.

        foreach ($chain as $rule) {
            // SF = nie jest RAW, nie jest finalne, from_ref != to_ref
            if (!empty($rule->is_raw)) {
                continue;
            }
            if (!empty($rule->is_final)) {
                continue;
            }
            if ($rule->from_ref === $rule->to_ref) {
                continue;
            }

            $sfRef = $rule->to_ref;

            // a) wyjścia spod SF wzdłuż main chain (już je narysowaliśmy wyżej)

            // b) boczne wyjścia z SF (np. alternatywne FG)
            //    Jeśli wszystkie wyjścia spod SF są finalnymi FG bez dalszych kroków,
            //    rysujemy je w jednej linii horyzontalnej:
            //    ↘ [FG20] [FG30] [FG40] ...
            //    W przeciwnym razie używamy starego trybu – każda gałąź osobno.
            if (!empty($sideOutputs[$sfRef])) {
                $outputsList = $sideOutputs[$sfRef];

                // Sprawdź, czy mamy prosty przypadek:
                //  - każde wyjście jest finalne (is_final)
                //  - brak dalszych bocznych wyjść z to_ref
                $allAreLeafFg = true;
                foreach ($outputsList as $rOut) {
                    if (empty($rOut->is_final)) {
                        $allAreLeafFg = false;
                        break;
                    }
                    if (!empty($sideOutputs[$rOut->to_ref])) {
                        $allAreLeafFg = false;
                        break;
                    }
                }

                if ($allAreLeafFg) {
                    // Sortujemy po step_order, żeby mieć 20,30,40,...
                    usort($outputsList, function ($a, $b) {
                        $sa = (int) $a->step_order;
                        $sb = (int) $b->step_order;
                        if ($sa === $sb) return strcmp($a->to_ref, $b->to_ref);
                        return $sa - $sb;
                    });

                    print '<div class="pr-flow-output-row">';
                    print '<span class="pr-diag-arrow">↘</span>';

                    foreach ($outputsList as $rOut) {
                        $typeOut    = self::detectNodeType($rOut, true); // boczne – FG
                        $tooltipOut = self::buildRuleTooltip($rOut, $typeOut, $rawRef);
                        print self::renderNode($rOut->to_ref, $typeOut, $rOut, $tooltipOut);
                    }

                    print '</div>'; // .pr-flow-output-row

                    // usuwamy, żeby nie renderować zaczepów spod SF drugi raz w pkt 3
                    unset($sideOutputs[$sfRef]);
                } else {
                    // STARY TRYB: każdą gałąź rysujemy jako pełną ścieżkę
                    foreach ($sideOutputs[$sfRef] as $r) {
                        $type    = self::detectNodeType($r, true); // boczne – FG
                        $tooltip = self::buildRuleTooltip($r, $type, $rawRef);

                        print '<div class="pr-flow-output-row">';
                        print '<span class="pr-diag-arrow">↘</span>';

                        // pierwszy box: FG (to_ref z reguły spod SF)
                        print self::renderNode($r->to_ref, $type, $r, $tooltip);

                        // Dociągamy ewentualne post-operacje z side_outputs[to_ref].
                        $currentFrom = $r->to_ref;
                        $visitedFrom = array();

                        while (!empty($sideOutputs[$currentFrom]) && empty($visitedFrom[$currentFrom])) {
                            $visitedFrom[$currentFrom] = true;

                            $nextList = $sideOutputs[$currentFrom];
                            $lastRule = null;

                            foreach ($nextList as $rNext) {
                                $typeN    = self::detectNodeType($rNext, true);
                                $tooltipN = self::buildRuleTooltip($rNext, $typeN, $rawRef);

                                print '<span class="pr-arrow">→</span>';
                                print self::renderNode($rNext->to_ref, $typeN, $rNext, $tooltipN);

                                $lastRule = $rNext;
                            }

                            // czy idziemy jeszcze dalej po to_ref ostatniej reguły
                            if ($lastRule && !empty($sideOutputs[$lastRule->to_ref]) && empty($visitedFrom[$lastRule->to_ref])) {
                                $currentFrom = $lastRule->to_ref;
                            } else {
                                break;
                            }
                        }

                        // Czyścimy zużyte boczne wyjścia, żeby nie rysować ich w pkt 3
                        foreach ($visitedFrom as $fromVisited => $_dummy) {
                            unset($sideOutputs[$fromVisited]);
                        }

                        print '</div>'; // .pr-flow-output-row
                    }

                    // usuwamy, żeby nie renderować zaczepów spod SF drugi raz w pkt 3
                    unset($sideOutputs[$sfRef]);
                }
            }
        }

        // ----------------------------------------------------------------
        // 3) Ewentualne inne boczne wyjścia (z innych etapów niż SF)
        // ----------------------------------------------------------------
        if (!empty($sideOutputs)) {
            // Szukamy takich from_ref, które same nie są to_ref innej bocznej reguły:
            // chcemy rysować tylko „korzenie” małych bocznych łańcuchów.
            //
            // Pomijamy samopętle (from_ref == to_ref), żeby gałęzie typu
            // FG2379-01 -> FG2379-01 były traktowane jako własny „korzeń”.
            $sideToRefs = array();
            foreach ($sideOutputs as $fromRefTmp => $rulesTmp) {
                foreach ($rulesTmp as $rTmp) {
                    if ($rTmp->to_ref !== $fromRefTmp) {
                        $sideToRefs[$rTmp->to_ref] = true;
                    }
                }
            }

            foreach ($sideOutputs as $fromRef => $rules) {
                // Jeśli from_ref występuje jako to_ref innej bocznej reguły,
                // to nie jest korzeniem – zostanie dociągnięty w innej linii.
                if (!empty($sideToRefs[$fromRef])) {
                    continue;
                }

                // Ustalenie typu node'a dla from_ref:
                //  - jeżeli to RAW – typ raw
                //  - w pozostałych przypadkach szukamy pierwszej reguły,
                //    dla której to_ref == from_ref i używamy detectNodeType()
                $fromType = 'op';
                if ($fromRef === $rawRef) {
                    $fromType = 'raw';
                } else {
                    foreach ($allRules as $arule) {
                        if ($arule->to_ref === $fromRef) {
                            $fromType = self::detectNodeType($arule);
                            break;
                        }
                    }
                }

                // Budujemy jeden wiersz bocznego łańcucha:
                //    ↘ [from_ref] → [kolejne to_ref...]
                print '<div class="pr-flow-output-row">';
                print '<span class="pr-diag-arrow">↘</span>';

                // pierwszy node
                print self::renderNode($fromRef, $fromType);

                // Dociągamy łańcuch po kolei
                $currentFrom = $fromRef;
                $visitedFrom = array();

                while (!empty($sideOutputs[$currentFrom]) && empty($visitedFrom[$currentFrom])) {
                    $visitedFrom[$currentFrom] = true;

                    $nextList = $sideOutputs[$currentFrom];
                    $lastRule = null;

                    foreach ($nextList as $rNext) {
                        $typeN    = self::detectNodeType($rNext, true);
                        $tooltipN = self::buildRuleTooltip($rNext, $typeN, $rawRef);

                        print '<span class="pr-arrow">→</span>';
                        print self::renderNode($rNext->to_ref, $typeN, $rNext, $tooltipN);

                        $lastRule = $rNext;
                    }

                    if ($lastRule && !empty($sideOutputs[$lastRule->to_ref]) && empty($visitedFrom[$lastRule->to_ref])) {
                        $currentFrom = $lastRule->to_ref;
                    } else {
                        break;
                    }
                }

                print '</div>'; // .pr-flow-output-row
            }
        }

        print '</div>'; // .pr-flow-container
    }

    /**
     * Render pojedynczego boxa (RAW / SF / FG / OP).
     *
     * @param string        $ref
     * @param string        $type  raw|sf|fg|op
     * @param stdClass|null $rule
     * @param string|null   $tooltipHtml
     * @return string
     */
    protected static function renderNode($ref, $type, $rule = null, $tooltipHtml = '')
    {
        $classes = array('pr-node', 'pr-flow-node');

        switch ($type) {
            case 'raw':
                $classes[] = 'pr-node-raw';
                break;
            case 'sf':
                $classes[] = 'pr-node-sf';
                break;
            case 'fg':
                $classes[] = 'pr-node-fg';
                break;
            default:
                $classes[] = 'pr-node-op';
        }

        $attrRuleId = '';
        if (!empty($rule->rowid)) {
            $attrRuleId = ' data-rule-id="'.((int) $rule->rowid).'"';
        }

        $out  = '<span class="'.implode(' ', $classes).'"'.$attrRuleId;

        if (!empty($tooltipHtml)) {
            $out .= ' title="'.dol_escape_htmltag($tooltipHtml).'"';
        }

        $out .= '>';

        $out .= dol_escape_htmltag($ref);

        if (!empty($rule->operation)) {
            $out .= ' <span class="pr-node-operation">'.dol_escape_htmltag($rule->operation).'</span>';
        }

        $out .= '</span>';

        return $out;
    }

    /**
     * Zbuduj tooltip dla RAW.
     *
     * @param string $rawRef
     * @return string
     */
    protected static function buildRawTooltip($rawRef)
    {
        global $langs;

        $lines = array();
        $lines[] = $langs->trans('RAW product').': '.$rawRef;

        return implode("\n", $lines);
    }

    /**
     * Zbuduj tooltip dla reguły (SF / FG / OP).
     *
     * @param stdClass $rule
     * @param string   $type
     * @param string   $rawRef
     * @return string
     */
    protected static function buildRuleTooltip($rule, $type, $rawRef)
    {
        global $langs;

        $lines = array();

        if (!empty($rule->operation)) {
            $lines[] = $langs->trans('Operation').': '.$rule->operation;
        }

        if (!empty($rule->workstation)) {
            $lines[] = $langs->trans('Workstation').': '.$rule->workstation;
        }

        $lines[] = 'From: '.$rule->from_ref;
        $lines[] = 'To: '.$rule->to_ref;
        $lines[] = 'Qty per from: '.$rule->qty_per_from;
        $lines[] = 'Step order: '.$rule->step_order;

        if (!empty($rule->is_final)) {
            $lines[] = '[FINAL FG]';
        }
        if (!empty($rule->is_raw)) {
            $lines[] = '[RAW]';
        }

        return implode("\n", $lines);
    }

    /**
     * Określ typ node'a na podstawie reguły.
     *
     * @param stdClass $rule
     * @param bool     $isSideOutput
     * @return string  raw|sf|fg|op
     */
    protected static function detectNodeType($rule, $isSideOutput = false)
    {
        // RAW (wejście) rozpoznajemy po is_raw
        if (!empty($rule->is_raw)) {
            return 'raw';
        }

        // boczne wyjście traktujemy jako FG (końcowy produkt)
        if ($isSideOutput) {
            return 'fg';
        }

        // finalne – FG
        if (!empty($rule->is_final)) {
            return 'fg';
        }

        // jeśli from != to => „proces” (SF / kolejny krok)
        if ($rule->from_ref !== $rule->to_ref) {
            return 'sf';
        }

        // post-operacja na tym samym produkcie
        return 'op';
    }
}
