<?php

/**
 * Build step-based flow for one ruleset.
 *
 * Wynik:
 *  - raw_ref:  kod materiału RAW (np. 62133)
 *  - chain:    lista reguł w głównej ścieżce, w kolejności step_order
 *  - side_outputs: [ from_ref => [ list of rules jako „boczne wyjścia” ] ]
 */

class RuleSetFlowBuilder
{
    /**
     * @param DoliDB $db
     * @param int    $ruleset_id
     * @return array
     *   [
     *     'raw_ref'      => string,
     *     'chain'        => array of rule objects,
     *     'side_outputs' => array from_ref => array(rule objects)
     *   ]
     */
    public static function build($db, $ruleset_id)
    {
        $out = array(
            'raw_ref'      => null,
            'chain'        => array(),
            'side_outputs' => array(),
            'rules'        => array(),
        );

        $rules = self::fetchRulesForRuleset($db, $ruleset_id);
        if (empty($rules)) {
            return $out;
        }

        // przechowujemy wszystkie aktywne reguły wprost w strukturze flow
        $out['rules'] = $rules;

        // przechowujemy wszystkie aktywne reguły wprost w strukturze flow
        $out['rules'] = $rules;

        // --- RAW: pierwszy is_raw=1, albo from_ref pierwszej reguły ---
        $raw_ref = null;
        foreach ($rules as $r) {
            if (!empty($r->is_raw)) {
                $raw_ref = $r->from_ref;
                break;
            }
        }
        if ($raw_ref === null) {
            $raw_ref = $rules[0]->from_ref;
        }
        $out['raw_ref'] = $raw_ref;

        // --- Mapa: from_ref => [rules]  oraz: czy dany produkt ma „dzieci” ---
        $rulesByFrom = array();
        $hasChildren = array(); // to_ref -> bool (czy występuje jako from_ref)

        foreach ($rules as $r) {
            $from = $r->from_ref;
            if (!isset($rulesByFrom[$from])) {
                $rulesByFrom[$from] = array();
            }
            $rulesByFrom[$from][] = $r;

            // zaznaczamy, że to_ref ma dzieci (pojawia się jako from_ref)
            // później uzupełnimy, gdy znamy wszystkie reguły
        }

        foreach ($rules as $r) {
            $from = $r->from_ref;
            $hasChildren[$from] = true;
        }

        // --- Budujemy główną ścieżkę ---
        $usedRowids   = array();
        $sideOutputs  = array();
        $chain        = array();

        self::buildMainChainRecursive(
            $raw_ref,
            $rulesByFrom,
            $hasChildren,
            $chain,
            $sideOutputs,
            $usedRowids
        );

        // --- Co zostało nieużyte, dorzucamy jako dodatkowe boczne wyjścia ---
        foreach ($rules as $r) {
            if (!empty($usedRowids[$r->rowid])) continue;

            $from = $r->from_ref;
            if (!isset($sideOutputs[$from])) {
                $sideOutputs[$from] = array();
            }
            $sideOutputs[$from][] = $r;
        }

        $out['chain']        = $chain;
        $out['side_outputs'] = $sideOutputs;

        return $out;
    }

    /**
     * Pobiera wszystkie reguły jednego rulesetu.
     *
     * @param DoliDB $db
     * @param int    $ruleset_id
     * @return array of stdClass
     */
    protected static function fetchRulesForRuleset($db, $ruleset_id)
    {
        $rules = array();

        $sql  = "SELECT rowid, from_ref, to_ref, operation, qty_per_from, workstation, ";
        $sql .= " step_order, is_final, is_raw ";
        $sql .= " FROM ".MAIN_DB_PREFIX."prod_rules";
        $sql .= " WHERE fk_ruleset = ".((int) $ruleset_id);
        $sql .= "   AND active = 1";
        $sql .= "   AND entity IN (".getEntity('prod_rules').")";
        $sql .= " ORDER BY step_order ASC, rowid ASC";

        $resql = $db->query($sql);
        if (!$resql) {
            dol_syslog(__METHOD__." SQL error: ".$db->lasterror(), LOG_ERR);
            return $rules;
        }

        while ($obj = $db->fetch_object($resql)) {
            $rules[] = $obj;
        }

        return $rules;
    }

    /**
     * Rekurencyjnie buduje główną ścieżkę:
     *  - wybiera regułę, która prowadzi dalej (jej to_ref ma dzieci) jako „main”
     *  - pozostałe reguły z tego samego from_ref -> side_outputs[from_ref][]
     *  - jeśli wszystkie reguły mają to_ref == from_ref -> traktujemy jako sekwencję post-operacji.
     *
     * @param string $currentRef
     * @param array  $rulesByFrom
     * @param array  $hasChildren
     * @param array  $chain           (by ref)
     * @param array  $sideOutputs     (by ref)
     * @param array  $usedRowids      (by ref)
     */
    protected static function buildMainChainRecursive(
        $currentRef,
        &$rulesByFrom,
        &$hasChildren,
        &$chain,
        &$sideOutputs,
        &$usedRowids
    ) {
        if (empty($rulesByFrom[$currentRef])) {
            return;
        }

        $list = $rulesByFrom[$currentRef];

        // Sortujemy po step_order, potem rowid
        usort($list, function ($a, $b) {
            if ((int) $a->step_order == (int) $b->step_order) {
                return ((int) $a->rowid < (int) $b->rowid) ? -1 : 1;
            }
            return ((int) $a->step_order < (int) $b->step_order) ? -1 : 1;
        });

        // sprawdzamy, czy wszystkie reguły mają to_ref == currentRef (post-operacje)
        $allSelf = true;
        foreach ($list as $r) {
            if ($r->to_ref !== $currentRef) {
                $allSelf = false;
                break;
            }
        }

        if ($allSelf) {
            // np. FG2378-01 -> FG2378-01 (CHEMI), FG2378-01 (PACKING)
            foreach ($list as $r) {
                $chain[] = $r;
                $usedRowids[$r->rowid] = true;
            }
            // nie przechodzimy dalej, bo byśmy zapętlili (to_ref == from_ref)
            return;
        }

        // Są różne to_ref – wybieramy „głównego” kandydata,
        // czyli taki, którego to_ref ma dzieci (jest from_ref dalej)
        $mainRule = null;
        foreach ($list as $r) {
            if (!empty($hasChildren[$r->to_ref])) {
                $mainRule = $r;
                break;
            }
        }

        // jeśli żaden nie prowadzi dalej – bierzemy po prostu pierwszy
        if ($mainRule === null) {
            $mainRule = $list[0];
        }

        // reszta idzie jako boczne wyjścia z currentRef
        foreach ($list as $r) {
            if ($r->rowid == $mainRule->rowid) continue;

            if (!isset($sideOutputs[$currentRef])) {
                $sideOutputs[$currentRef] = array();
            }
            $sideOutputs[$currentRef][] = $r;
            $usedRowids[$r->rowid] = true;
        }

        // dodajemy mainRule do łańcucha
        $chain[] = $mainRule;
        $usedRowids[$mainRule->rowid] = true;

        // idziemy dalej po to_ref
        self::buildMainChainRecursive(
            $mainRule->to_ref,
            $rulesByFrom,
            $hasChildren,
            $chain,
            $sideOutputs,
            $usedRowids
        );
    }
}
