<?php

/**
 * Render flow for one RuleSet.
 *
 * Oczekuje struktury z RuleSetFlowBuilder::build():
 *  - raw_ref      : string
 *  - chain        : array of rule objects (główna ścieżka, po step_order)
 *  - side_outputs : array from_ref => array(rule objects)  (boczne FG/SCRAP)
 */

class RuleSetFlowRenderer
{
    /**
     * @param array $flow Wynik z RuleSetFlowBuilder::build()
     */
    public static function render($flow)
    {
        global $langs;

        if (empty($flow) || empty($flow['raw_ref'])) {
            print '<br>';
            print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');
            print '<div class="pr-flow-container">';
            print '<span class="opacitymedium">No rules in this RuleSet.</span>';
            print '</div>';
            return;
        }

        $rawRef      = $flow['raw_ref'];
        $chain       = $flow['chain'];        // główna ścieżka (po step_order)
        $sideOutputs = $flow['side_outputs']; // boczne wyjścia

        // Zbiór wszystkich reguł (główna ścieżka + boczne) – przyda się
        // do określania typu node'a dla from_ref w bocznych gałęziach
        $allRules = $chain;
        foreach ($sideOutputs as $tmpFrom => $tmpRules) {
            foreach ($tmpRules as $tmpR) {
                $allRules[] = $tmpR;
            }
        }

        print '<br>';
        print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');

        // Tickbox do pokazywania komentarza (operation) obok ref
        print '<div class="pr-flow-options">';
        print '<label><input type="checkbox" id="pr-show-comments"> Show comment</label>';
        print '</div>';

        print '<div class="pr-flow-container">';

        // ----------------------------------------------------------------
        // 1) LINIA GŁÓWNA: RAW → SF (tylko pierwszy krok z chain)
        // ----------------------------------------------------------------
        print '<div class="pr-flow-row">';

        // RAW
        $tooltipRaw = self::buildRawTooltip($rawRef);
        print self::renderNode($rawRef, 'raw', null, $tooltipRaw);

        $firstRule = null;
        if (!empty($chain)) {
            $firstRule = $chain[0];

            // RAW → SF
            print '<span class="pr-arrow">→</span>';

            $typeFirst    = self::detectNodeType($firstRule);
            $tooltipFirst = self::buildRuleTooltip($firstRule, $typeFirst, $rawRef);

            print self::renderNode($firstRule->to_ref, $typeFirst, $firstRule, $tooltipFirst);
        }

        print '</div>'; // .pr-flow-row

        // ----------------------------------------------------------------
        // 2) PIĘTRO NIŻEJ POD SF:
        //    ↘ FG2378-01 (TRIM) → FG2378-01 (CHEMI) → FG2378-01 (PACKING)
        //    ↘ FG2379-01 (TRIM)
        // ----------------------------------------------------------------
        print '<div class="pr-flow-outputs">';

        if ($firstRule) {
            $sfRef = $firstRule->to_ref;

            // a) główna ścieżka po SF: chain[1..]
            $restChain = array_slice($chain, 1);

            if (!empty($restChain)) {
                print '<div class="pr-flow-output-row">';

                // start spod SF
                print '<span class="pr-diag-arrow">↘</span>';

                $firstUnder = true;
                foreach ($restChain as $r) {
                    $type    = self::detectNodeType($r);
                    $tooltip = self::buildRuleTooltip($r, $type, $rawRef);

                    // pierwszy node: bez strzałki z lewej (już jest ↘),
                    // kolejne: z poziomą strzałką →
                    if (!$firstUnder) {
                        print '<span class="pr-arrow">→</span>';
                    }
                    $firstUnder = false;

                    print self::renderNode($r->to_ref, $type, $r, $tooltip);
                }

                print '</div>'; // .pr-flow-output-row
            }

            // b) boczne wyjścia z SF (np. alternatywne FG)
            //    Każdą z nich rysujemy jako pełną gałąź:
            //    ↘ [FGx] → [post‑operacje FGx] ...
            if (!empty($sideOutputs[$sfRef])) {
                foreach ($sideOutputs[$sfRef] as $r) {
                    $type    = self::detectNodeType($r, true); // boczne – FG
                    $tooltip = self::buildRuleTooltip($r, $type, $rawRef);

                    print '<div class="pr-flow-output-row">';
                    print '<span class="pr-diag-arrow">↘</span>';

                    // pierwszy box: FG (to_ref z reguły spod SF)
                    print self::renderNode($r->to_ref, $type, $r, $tooltip);

                    // Dociągamy ewentualne post‑operacje z side_outputs[to_ref].
                    $currentFrom = $r->to_ref;
                    $visitedFrom = array();

                    while (!empty($sideOutputs[$currentFrom]) && empty($visitedFrom[$currentFrom])) {
                        $visitedFrom[$currentFrom] = true;

                        $nextList = $sideOutputs[$currentFrom];
                        $lastRule = null;

                        foreach ($nextList as $rNext) {
                            $typeN    = self::detectNodeType($rNext, true);
                            $tooltipN = self::buildRuleTooltip($rNext, $typeN, $rawRef);

                            print '<span class="pr-arrow">→</span>';
                            print self::renderNode($rNext->to_ref, $typeN, $rNext, $tooltipN);

                            $lastRule = $rNext;
                        }

                        // czy idziemy jeszcze dalej po to_ref ostatniej reguły
                        if ($lastRule && !empty($sideOutputs[$lastRule->to_ref]) && empty($visitedFrom[$lastRule->to_ref])) {
                            $currentFrom = $lastRule->to_ref;
                        } else {
                            break;
                        }
                    }

                    // Czyścimy zużyte boczne wyjścia, żeby nie rysować ich w pkt 3
                    foreach ($visitedFrom as $fromVisited => $_dummy) {
                        unset($sideOutputs[$fromVisited]);
                    }

                    print '</div>'; // .pr-flow-output-row
                }

                // usuwamy, żeby nie renderować zaczepów spod SF drugi raz w pkt 3
                unset($sideOutputs[$sfRef]);
            }
        }

        
        // ----------------------------------------------------------------
        // 3) Ewentualne inne boczne wyjścia (z innych etapów niź SF)
        //     Chcemy, żeby cała boczna gałąź (np. FG2379-01 + jego
        //     post‑operacje) była narysowana w JEDNEJ linii:
        //     ↘ [from_ref] → [to_ref1] → [to_ref2] ...
        //
        //     Dlatego:
        //      - najpierw znajdujemy „korzenie” gałęzi (from_ref, które
        //        nie występują jako to_ref w innych bocznych regułach),
        //      - dla każdego takiego korzenia budujemy łańcuch, dociągając
        //        kolejne kroki z side_outputs[to_ref].
        // ----------------------------------------------------------------
        if (!empty($sideOutputs)) {
            // Zbierz wszystkie to_ref z bocznych reguł.
            // Pomijamy samopętle (from_ref == to_ref), żeby gałęzie typu
            // FG2379-01 -> FG2379-01 były traktowane jako własny „korzeń”.
            $sideToRefs = array();
            foreach ($sideOutputs as $fromRefTmp => $rulesTmp) {
                foreach ($rulesTmp as $rTmp) {
                    if ($rTmp->to_ref !== $fromRefTmp) {
                        $sideToRefs[$rTmp->to_ref] = true;
                    }
                }
            }

            foreach ($sideOutputs as $fromRef => $rules) {
                // Jeśli from_ref występuje jako to_ref innej bocznej reguły,
                // to nie jest korzeniem – zostanie dociągnięty w innej linii.
                if (!empty($sideToRefs[$fromRef])) {
                    continue;
                }

                // Ustalenie typu node'a dla from_ref:
                //  - jeżeli to RAW – typ raw
                //  - w pozostałych przypadkach szukamy pierwszej reguły,
                //    dla której to_ref == from_ref i używamy detectNodeType()
                $fromType = 'op';
                if ($fromRef === $rawRef) {
                    $fromType = 'raw';
                } else {
                    foreach ($allRules as $arule) {
                        if ($arule->to_ref === $fromRef) {
                            $fromType = self::detectNodeType($arule);
                            break;
                        }
                    }
                }

                // Z jednego from_ref może wychodzić kilka bocznych linii,
                // więc każdą regułę startową rysujemy jako osobny rząd.
                foreach ($rules as $startRule) {
                    print '<div class="pr-flow-output-row">';
                    print '<span class="pr-diag-arrow">↘</span>';

                    // pierwszy box: from_ref
                    print self::renderNode($fromRef, $fromType, null, '');

                    // budujemy łańcuch począwszy od startRule
                    $currentRule = $startRule;
                    $visitedFrom = array();

                    while ($currentRule !== null) {
                        $type    = self::detectNodeType($currentRule, true);
                        $tooltip = self::buildRuleTooltip($currentRule, $type, $rawRef);

                        print '<span class="pr-arrow">→</span>';
                        print self::renderNode($currentRule->to_ref, $type, $currentRule, $tooltip);

                        $nextFrom = $currentRule->to_ref;

                        // Samopętla (from_ref == to_ref) – rysujemy tylko raz
                        // i kończymy gałąź, żeby nie dublować boxa.
                        if ($nextFrom === $currentRule->from_ref) {
                            $currentRule = null;
                            break;
                        }

                        // jeśli nie mamy dalszych bocznych reguł z nextFrom
                        // albo już je wykorzystaliśmy w tej linii – kończymy.
                        if (empty($sideOutputs[$nextFrom]) || !empty($visitedFrom[$nextFrom])) {
                            $currentRule = null;
                            break;
                        }

                        $visitedFrom[$nextFrom] = true;

                        // wszystkie reguły z nextFrom traktujemy jako kolejne kroki
                        // w tej samej linii (typowa sytuacja: post‑operacje FG).
                        $nextList = $sideOutputs[$nextFrom];
                        $currentRule = null; // ustawimy na ostatnią z nextList

                        foreach ($nextList as $rNext) {
                            $typeN    = self::detectNodeType($rNext, true);
                            $tooltipN = self::buildRuleTooltip($rNext, $typeN, $rawRef);

                            print '<span class="pr-arrow">→</span>';
                            print self::renderNode($rNext->to_ref, $typeN, $rNext, $tooltipN);

                            $currentRule = $rNext;
                        }

                        // pętla while spróbuje ewentualnie dociągnąć kolejne poziomy
                        // (side_outputs z nowego to_ref), o ile istnieją
                    }

                    print '</div>'; // .pr-flow-output-row
                }
            }
        }

        print '</div>'; // .pr-flow-outputs
        print '</div>'; // .pr-flow-container
    }

    /**
     * Render pojedynczego boxa (RAW / SF / FG / OP).
     *
     * @param string        $ref
     * @param string        $type  raw|sf|fg|op
     * @param stdClass|null $rule
     * @param string|null   $tooltipHtml
     * @return string
     */
    protected static function renderNode($ref, $type, $rule = null, $tooltipHtml = '')
    {
        $classes = array('pr-node', 'pr-flow-node');

        switch ($type) {
            case 'raw':
                $classes[] = 'pr-node-raw';
                break;
            case 'sf':
                $classes[] = 'pr-node-sf';
                break;
            case 'fg':
                $classes[] = 'pr-node-fg';
                break;
            default:
                $classes[] = 'pr-node-op';
                break;
        }

        // Tooltip: używamy atrybutu data-tooltip-html, bo tego oczekuje JS
        $attrTooltip = '';
        if ($tooltipHtml !== '') {
            // Każdą linię już escapowaliśmy, więc tutaj tylko " żeby atrybut był poprawny
            $safe = str_replace('"', '&quot;', $tooltipHtml);
            // Ustaw oba atrybuty, bo różne wersje JS mogą czytać różne
            $attrTooltip = ' data-tooltip-html="'.$safe.'" data-tooltip="'.$safe.'"';
        }

        // COMMENT = operation w nawiasie (później sterowane checkboxem Show comment)
        $commentHtml = '';
        if ($rule && !empty($rule->operation)) {
            $commentHtml = ' <span class="pr-node-comment">('
                .dol_escape_htmltag($rule->operation)
                .')</span>';
        }

        // Id reguły – potrzebne np. do kasowania po kliknięciu node'a w JS
        $attrRuleId = '';
        if ($rule && !empty($rule->rowid)) {
            $attrRuleId = ' data-rule-id="'.((int) $rule->rowid).'"';
        }

        return '<span class="'.implode(' ', $classes).'"'.$attrTooltip.$attrRuleId.'">'
            .dol_escape_htmltag($ref)
            .$commentHtml
            .'</span>';
    }

    /**
     * Prosty tooltip dla RAW.
     */
    protected static function buildRawTooltip($rawRef)
    {
        $lines   = array();
        $lines[] = 'RAW material: '.$rawRef;

        $safeLines = array();
        foreach ($lines as $l) {
            $safeLines[] = dol_escape_htmltag($l);
        }

        return implode('<br>', $safeLines);
    }

    /**
     * Buduje tooltip dla reguły.
     *
     * @param stdClass $rule
     * @param string   $type
     * @param string   $rawRef
     * @return string
     */
    protected static function buildRuleTooltip($rule, $type, $rawRef)
    {
        $lines = array();

        if ($type === 'raw') {
            $lines[] = 'RAW material: '.$rawRef;
        }

        $lines[] = 'From: '.$rule->from_ref;
        $lines[] = 'To: '.$rule->to_ref;

        if (!empty($rule->operation)) {
            $lines[] = 'Operation: '.$rule->operation;
        }
        if (!empty($rule->workstation)) {
            $lines[] = 'Workstation: '.$rule->workstation;
        }

        // Qty per from (ładnie: 1, 0.5, 0.25 itd.)
        if (isset($rule->qty_per_from)) {
            $qty     = (float) $rule->qty_per_from;
            $qty_fmt = rtrim(rtrim(number_format($qty, 4, '.', ''), '0'), '.');
            $lines[] = 'Qty per from: '.$qty_fmt;
        }

        if (!empty($rule->step_order)) {
            $lines[] = 'Step order: '.$rule->step_order;
        }

        if (!empty($rule->is_final)) {
            $lines[] = 'Final (FG)';
        }

        $safeLines = array();
        foreach ($lines as $l) {
            $safeLines[] = dol_escape_htmltag($l);
        }

        return implode('<br>', $safeLines);
    }

    /**
     * Określa typ node'a: raw/sf/fg/op
     *
     * @param stdClass $rule
     * @param bool     $isSideOutput  czy to boczne wyjście
     * @return string
     */
    protected static function detectNodeType($rule, $isSideOutput = false)
    {
        // boczne wyjścia prawie zawsze FG
        if ($isSideOutput) {
            return 'fg';
        }

        // finalne – FG
        if (!empty($rule->is_final)) {
            return 'fg';
        }

        // jeśli from != to => „proces” (SF / kolejny krok)
        if ($rule->from_ref !== $rule->to_ref) {
            return 'sf';
        }

        // post-operacja na tym samym produkcie
        return 'op';
    }
}
