<?php

/**
 * Render flow for one RuleSet.
 *
 * Oczekuje struktury z RuleSetFlowBuilder::build():
 *  - raw_ref      : string
 *  - chain        : array of rule objects (główna ścieżka, po step_order)
 *  - side_outputs : array from_ref => array(rule objects)  (boczne FG/SCRAP)
 */

class RuleSetFlowRenderer
{
    /**
     * @param array $flow Wynik z RuleSetFlowBuilder::build()
     */
    public static function render($flow)
    {
        global $langs;

        if (empty($flow) || empty($flow['raw_ref'])) {
            print '<br>';
            print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');
            print '<div class="pr-flow-container">';
            print '<span class="opacitymedium">No rules in this RuleSet.</span>';
            print '</div>';
            return;
        }

        // Ustalamy bazowy SF (rawRef)
        $rawRef      = !empty($flow['raw_ref']) ? $flow['raw_ref'] : '';
        $chain       = isset($flow['chain']) ? $flow['chain'] : array();        // nieużywane w prostym routingu
        $sideOutputs = isset($flow['side_outputs']) ? $flow['side_outputs'] : array(); // nieużywane
        $rules       = isset($flow['rules']) ? $flow['rules'] : array();        // wszystkie aktywne reguły

        // Jeżeli builder nie podał raw_ref, bierzemy z pierwszej reguły
        if (empty($rawRef) && !empty($rules)) {
            $first = reset($rules);
            if (!empty($first->from_ref)) {
                $rawRef = $first->from_ref;
            }
        }

        print '<br>';
        print load_fiche_titre('Flow (horizontal) for selected RuleSet', '', 'generic');

        // Tickbox do pokazywania komentarza (operation) obok ref
        print '<div class="pr-flow-options">';
        print '<label><input type="checkbox" id="pr-show-comments"> Show comment</label>';
        print '</div>';

        print '<div class="pr-flow-container">';

                // ----------------------------------------------------------------
        // 1) GŁÓWNA LINIA: SF (RuleSet ref)
        // ----------------------------------------------------------------
        print '<div class="pr-flow-row">';
        // SF zawsze bez tooltipa – proces jest pokazany w gałęziach
        print self::renderNode($rawRef, 'sf', null, '');
        print '</div>'; // .pr-flow-row

        // ----------------------------------------------------------------
        // 2) GAŁĘZIE DLA KAŻDEGO FG (grupowanie po to_ref, sortowanie po step_order)
        // ----------------------------------------------------------------

        // Grupujemy wszystkie reguły wg to_ref (FG/SF docelowego)
        $rulesByTo = array();
        foreach ($rules as $r) {
            if (empty($r->to_ref)) continue;
            if (!isset($rulesByTo[$r->to_ref])) {
                $rulesByTo[$r->to_ref] = array();
            }
            $rulesByTo[$r->to_ref][] = $r;
        }

        // Dla każdej części docelowej (FG/SF) budujemy jedną gałąź:
        //     SF
        //        ↘ FG → FG → FG  (kolejne kroki procesu po step_order)
        foreach ($rulesByTo as $toRef => $list) {
            // sortujemy po step_order, potem po rowid
            usort($list, function ($a, $b) {
                $sa = (int) $a->step_order;
                $sb = (int) $b->step_order;
                if ($sa === $sb) {
                    if ($a->rowid == $b->rowid) return 0;
                    return ($a->rowid < $b->rowid) ? -1 : 1;
                }
                return ($sa < $sb) ? -1 : 1;
            });

            print '<div class="pr-flow-output-row">';
            print '<span class="pr-diag-arrow">↘</span>';

            $first = true;
            foreach ($list as $rule) {
                $type    = 'fg';
                $tooltip = self::buildRuleTooltip($rule, $type, $rawRef);

                if (!$first) {
                    print '<span class="pr-arrow">→</span>';
                }
                $first = false;

                // Każdy krok to ten sam FG (to_ref), ale kolejny etap procesu
                print self::renderNode($rule->to_ref, $type, $rule, $tooltip);
            }

            print '</div>'; // .pr-flow-output-row
        }

        // Zamykamy kontener flow
        print '</div>'; // .pr-flow-container
    }

    /**
     * Renderuje pojedynczy node w flow (SF/FG).
     *
     * @param string      $ref          Kod produktu
     * @param string      $type         'sf' lub 'fg'
     * @param stdClass    $rule         Powiązana reguła (opcjonalnie)
     * @param string|null $tooltipHtml  Treść tooltipa (HTML)
     * @return string
     */
    protected static function renderNode($ref, $type = 'sf', $rule = null, $tooltipHtml = '')
    {
        if ($ref === null || $ref === '') {
            return '';
        }

        $classes = array('pr-node', 'pr-flow-node');
        if ($type === 'sf') {
            $classes[] = 'pr-node-raw';
        } else {
            $classes[] = 'pr-node-fg';
        }

        $out = '<span class="'.implode(' ', $classes).'"';

        if (!empty($rule->rowid)) {
            $out .= ' data-rule-id="'.(int) $rule->rowid.'"';
        }

        if (!empty($tooltipHtml)) {
            // używamy data-tooltip-html, bo na to patrzy JS od dymków
            $out .= ' data-tooltip-html="'.dol_escape_htmltag($tooltipHtml).'"';
        }

        $out .= '>';

        // pokazujemy tylko kod do pierwszej spacji (SF/FG bez dalszego opisu)
        $displayRef = $ref;
        if (preg_match('/^(\S+)/', (string) $ref, $m)) {
            $displayRef = $m[1];
        }
        $out .= dol_escape_htmltag($displayRef);

        if (!empty($rule) && !empty($rule->operation)) {
            // komentarz operacji – sterowany checkboxem "Show comment"
            $out .= ' <span class="pr-node-comment">('.dol_escape_htmltag($rule->operation).')</span>';
        }

        $out .= '</span>';

        return $out;
    }

    /**
     * Buduje HTML tooltipa dla reguły.
     *
     * @param stdClass $rule
     * @param string   $type
     * @param string   $rawRef
     * @return string
     */
    protected static function buildRuleTooltip($rule, $type, $rawRef)
    {
        global $langs;

        $lines = array();

        if (!empty($rule->operation)) {
            $lines[] = $langs->transnoentitiesnoconv('Operation') . ': ' . $rule->operation;
        }
        if (!empty($rule->workstation)) {
            $lines[] = $langs->transnoentitiesnoconv('Workstation') . ': ' . $rule->workstation;
        }
        if (!empty($rule->qty_per_from)) {
            $lines[] = $langs->transnoentitiesnoconv('QtyPerFrom') . ': ' . price($rule->qty_per_from, 0, $langs, 0, 0);
        }
        if (!empty($rule->step_order)) {
            $lines[] = $langs->transnoentitiesnoconv('Step Order') . ': ' . $rule->step_order;
        }
        if (!empty($rule->from_ref) || !empty($rule->to_ref)) {
            $lines[] = 'From: ' . dol_escape_htmltag($rule->from_ref) . ' → To: ' . dol_escape_htmltag($rule->to_ref);
        }
        if (!empty($rule->note)) {
            $lines[] = $langs->transnoentitiesnoconv('Note') . ': ' . dol_escape_htmltag($rule->note);
        }

        return implode('<br>', $lines);
    }
}
