/**
 * Planning Timeline - Read-only visualization
 */

(function() {
  'use strict';

  var config = {};
  var timelineData = null;
  var selectedJob = null;
  var editMode = false;
  var editOriginalData = null; // Stores original values for cancel
  var showPastMode = false; // Toggle state for Past/Today view

  // Fetch state management to prevent overlapping requests
  var isFetching = false;
  var pendingFetch = false;
  var fetchAbortController = null;
  var fetchDebounceTimer = null;
  var lastFetchTime = 0;

  // Debug flag (enabled by ?debug=1 or window.PLANNING_DEBUG)
  var debugEnabled = function() {
    if (window.PLANNING_DEBUG === true) return true;
    if (typeof URLSearchParams !== 'undefined') {
      try {
        return new URLSearchParams(window.location.search).has('debug');
      } catch (e) { return false; }
    }
    return false;
  }();

  // DOM helpers
  function $(id) { return document.getElementById(id); }

  /**
   * Compute day column width based on container and day count.
   * Returns clamped value: min 55px, max 110px.
   * @param {number} dayCount Number of days
   * @param {number} containerWidth Container width in pixels (optional)
   * @returns {number} Day width in pixels
   */
  function getDayWidth(dayCount, containerWidth) {
    var machineColWidth = 140;
    var defaultWidth = 80;
    if (!containerWidth || containerWidth <= 0) {
      containerWidth = 800; // fallback
    }
    if (!dayCount || dayCount <= 0) {
      return defaultWidth;
    }
    var availableWidth = containerWidth - machineColWidth - 20; // scrollbar buffer
    var computedWidth = Math.floor(availableWidth / dayCount);
    return Math.max(55, Math.min(computedWidth, 110));
  }

  /**
   * Read config from data attributes on #plTimelinePage
   */
  function loadConfig() {
    var el = $('plTimelinePage');
    if (!el) return false;
    config.group = el.dataset.group || '';
    config.days = parseInt(el.dataset.days, 10) || 14;
    config.endpoint = el.dataset.endpoint || '';
    config.addjobEndpoint = el.dataset.addjobEndpoint || '';
    config.updatejobEndpoint = el.dataset.updatejobEndpoint || '';
    config.deletejobEndpoint = el.dataset.deletejobEndpoint || '';
    config.token = el.dataset.token || '';
    config.canAddjob = el.dataset.canAddjob === '1';
    config.canEditjob = el.dataset.canEditjob === '1';
    config.canDeletejob = el.dataset.canDeletejob === '1';
    config.ganttMode = parseInt(el.dataset.ganttMode, 10) || 1;
    config.searchProductsEndpoint = el.dataset.searchProductsEndpoint || '';
    try {
      config.workstations = JSON.parse(el.dataset.workstations || '[]');
    } catch (e) {
      config.workstations = [];
    }
    return !!config.endpoint;
  }
  function esc(s) { return String(s).replace(/[&<>"']/g, function(c) { return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]; }); }

  /**
   * Calculate offset in days from a given date to today.
   * @param {string} dateStr Date in YYYY-MM-DD format
   * @returns {number} Offset: 0 = today, -1 = yesterday, 1 = tomorrow
   */
  function getDayOffset(dateStr) {
    var parts = dateStr.split('-');
    var targetDate = new Date(parseInt(parts[0], 10), parseInt(parts[1], 10) - 1, parseInt(parts[2], 10));
    var today = new Date();
    today.setHours(0, 0, 0, 0);
    targetDate.setHours(0, 0, 0, 0);
    return Math.floor((targetDate - today) / (1000 * 60 * 60 * 24));
  }

  /**
   * Fetch timeline data from endpoint with in-flight guard and debounce
   */
  function fetchTimeline() {
    // Debounce rapid calls (200ms minimum between fetches)
    if (Date.now() - lastFetchTime < 200) {
      pendingFetch = true;
      clearTimeout(fetchDebounceTimer);
      fetchDebounceTimer = setTimeout(function() {
        if (pendingFetch && !isFetching) {
          pendingFetch = false;
          fetchTimeline();
        }
      }, 200);
      return;
    }

    // Prevent overlapping fetch requests
    if (isFetching) {
      pendingFetch = true;
      return;
    }

    // Abort any previous in-flight request
    if (fetchAbortController) {
      fetchAbortController.abort();
    }
    fetchAbortController = new AbortController();

    isFetching = true;
    pendingFetch = false;
    lastFetchTime = Date.now();

    var startDate = getTodayDate();
    if (showPastMode) {
      var d = new Date();
      d.setDate(d.getDate() - config.days);
      startDate = d.getFullYear() + '-' +
                  String(d.getMonth() + 1).padStart(2, '0') + '-' +
                  String(d.getDate()).padStart(2, '0');
    }
    var url = config.endpoint + '?group=' + encodeURIComponent(config.group) +
              '&days=' + config.days + '&start=' + startDate;

    var grid = $('plTimelineGrid');
    if (grid) grid.innerHTML = '<div class="pl-timeline-loading">Loading...</div>';

    fetch(url, { credentials: 'same-origin', signal: fetchAbortController.signal })
      .then(function(r) {
        if (!r.ok) {
          return r.text().then(function(text) {
            throw new Error('Server error: ' + r.status + ' - ' + (text ? text.substring(0, 100) : 'no response'));
          });
        }
        return r.json().catch(function(err) {
          throw new Error('Invalid JSON response: ' + err.message);
        });
      })
      .then(function(data) {
        if (data.error) {
          renderError(data.error);
          return;
        }
        timelineData = data;
        renderTimeline(data);
      })
      .catch(function(err) {
        if (err.name !== 'AbortError') {
          renderError('Failed to load timeline: ' + err.message);
        }
      })
      .finally(function() {
        isFetching = false;
        if (pendingFetch) {
          pendingFetch = false;
          fetchTimeline();
        }
      });
  }

  /**
   * Get today's date as YYYY-MM-DD
   */
  function getTodayDate() {
    var d = new Date();
    return d.getFullYear() + '-' +
           String(d.getMonth() + 1).padStart(2, '0') + '-' +
           String(d.getDate()).padStart(2, '0');
  }

  /**
   * Format date for display
   */
  function formatDate(dateStr) {
    if (!dateStr) return '';
    var parts = dateStr.split('-');
    return parts[2] + '/' + parts[1];
  }

  /**
   * Render error message
   */
  function renderError(msg) {
    var grid = $('plTimelineGrid');
    if (grid) {
      grid.innerHTML = '<div class="pl-timeline-error">' + esc(msg) + '</div>';
    }
  }

  /**
   * Render the timeline grid
   */
  function renderTimeline(data) {
    var grid = $('plTimelineGrid');
    if (!grid) return;

    // Set CSS variables for consistent column sizing
    var dayCount = data.planning_days ? data.planning_days.length : data.days_limit;

    // Compute day width using helper function
    var panel = $('plTimelinePanel');
    var containerWidth = panel ? panel.clientWidth : 800;
    var dayWidth = getDayWidth(dayCount, containerWidth);

    grid.style.setProperty('--pl-day-count', dayCount);
    grid.style.setProperty('--pl-day-width', dayWidth + 'px');

    // Toggle compact mode for narrow columns (under 80px)
    grid.classList.toggle('pl-compact', dayWidth < 80);

    // Clear previous content completely
    grid.innerHTML = '';

    var html = '';

    // Day header row - use planning_days as single source of truth
    var planningDays = data.planning_days || [];
    html += '<div class="pl-tl-row pl-tl-header-row">';
    html += '<div class="pl-tl-machine-cell pl-tl-header-machine">Machine</div>';
    for (var i = 0; i < planningDays.length; i++) {
      var dayInfo = planningDays[i];
      var dayOffset = getDayOffset(dayInfo.date);
      var dayLabel;
      if (dayOffset === 0) {
        dayLabel = 'Today';
      } else if (dayOffset < 0) {
        dayLabel = 'D' + dayOffset; // e.g., D-5
      } else {
        dayLabel = 'D+' + dayOffset; // e.g., D+5
      }
      var dateLabel = formatDate(dayInfo.date);
      html += '<div class="pl-tl-day-cell pl-tl-header-day" data-day="' + i + '">';
      html += '<div class="pl-tl-day-label">' + dayLabel + '</div>';
      html += '<div class="pl-tl-day-date">' + dateLabel + '</div>';
      html += '</div>';
    }
    html += '</div>'; // End header row

    // Machine rows
    if (!data.machines || data.machines.length === 0) {
      html += '<div class="pl-tl-empty">No machines found for this group.</div>';
    } else {
      data.machines.forEach(function(machine) {
        // Choose renderer based on config.ganttMode
        var rowHtml = config.ganttMode === 1
          ? renderMachineRowGantt(machine, planningDays, dayWidth)
          : renderMachineRow(machine, planningDays, dayWidth);
        html += rowHtml;
      });
    }

    grid.innerHTML = html;

    // Bind click events for job blocks (both capacity and gantt modes)
    grid.querySelectorAll('.pl-tl-job-block').forEach(function(block) {
      block.addEventListener('click', function(e) {
        e.stopPropagation(); // Prevent cell click
        selectJob(block.dataset.machineId, block.dataset.jobId);
      });
    });

    // Bind click events for gantt bars
    grid.querySelectorAll('.pl-tl-job-gantt-bar').forEach(function(bar) {
      bar.addEventListener('click', function(e) {
        e.stopPropagation();
        selectJob(bar.dataset.machineId, bar.dataset.jobId);
      });
    });

    // Bind click events for day cells (add job on empty cell click)
    // Works in both capacity and gantt modes
    if (config.canAddjob) {
      grid.querySelectorAll('.pl-tl-machine-row .pl-tl-day-cell').forEach(function(cell) {
        cell.addEventListener('click', function() {
          var machineId = cell.dataset.machineId;
          var date = cell.dataset.date;
          if (machineId && date) {
            // Get workstation label from row
            var row = cell.closest('.pl-tl-machine-row');
            var machineLabel = '';
            if (row) {
              var labelEl = row.querySelector('.pl-tl-machine-label');
              var refEl = row.querySelector('.pl-tl-machine-ref');
              var lblTxt = (labelEl && labelEl.textContent) ? labelEl.textContent.trim() : '';
              var refTxt = (refEl && refEl.textContent) ? refEl.textContent.trim() : '';
              if (lblTxt && refTxt && lblTxt !== refTxt) {
                machineLabel = lblTxt + ' - ' + refTxt;
              } else if (lblTxt) {
                machineLabel = lblTxt;
              } else if (refTxt) {
                machineLabel = refTxt;
              }
            }
            openAddJobModalWithPrefill(machineId, machineLabel, date);
          }
        });
      });
    }
  }

  /**
   * Render a single machine row
   * @param {Object} machine Machine data
   * @param {Array} planningDays Planning days array
   * @param {number} dayWidth Day column width in pixels
   */
  function renderMachineRow(machine, planningDays, dayWidth) {
    var html = '<div class="pl-tl-row pl-tl-machine-row" data-machine-id="' + machine.id + '">';

    // Machine label cell
    html += '<div class="pl-tl-machine-cell">';
    html += '<div class="pl-tl-machine-ref">' + esc(machine.ref) + '</div>';
    if (machine.label && machine.label !== machine.ref) {
      html += '<div class="pl-tl-machine-label">' + esc(machine.label) + '</div>';
    }
    html += '</div>';

    // Day cells with job segments - use planningDays.length as source of truth
    for (var dayIdx = 0; dayIdx < planningDays.length; dayIdx++) {
      var dayData = machine.days[dayIdx];
      var availableHours = dayData ? dayData.available_hours : 0;
      var maxHours = dayData ? dayData.effective_capacity : 24;
      var dayDate = planningDays[dayIdx] ? planningDays[dayIdx].date : '';

      html += '<div class="pl-tl-day-cell" data-day="' + dayIdx + '" data-date="' + dayDate + '" data-machine-id="' + machine.id + '">';

      // Find job segments for this day
      var segmentsForDay = [];
      machine.jobs.forEach(function(job) {
        job.segments.forEach(function(seg) {
          // Only render segments for this specific day
          // Skip off-timeline segments (day_index === -1) to prevent leaking into wrong view
          if (seg.day_index === dayIdx) {
            segmentsForDay.push({ job: job, segment: seg });
          }
        });
      });

      // Render job blocks
      segmentsForDay.forEach(function(item) {
        var job = item.job;
        var seg = item.segment;
        // Compute width in pixels: (hours / capacity) * dayWidth, clamped
        var ratio = maxHours > 0 ? (seg.hours / maxHours) : 0;
        var rawWidth = ratio * dayWidth;
        var blockWidth = Math.max(6, Math.min(rawWidth, dayWidth - 8));

        html += '<div class="pl-tl-job-block" ';
        html += 'data-machine-id="' + machine.id + '" ';
        html += 'data-job-id="' + job.id + '" ';
        html += 'style="width: ' + blockWidth.toFixed(0) + 'px;" ';
        var jobLabel = job.works_order_no || job.fg || ('Job#' + job.id);
        html += 'title="' + esc(jobLabel) + ' - ' + seg.hours.toFixed(1) + 'h / ' + maxHours.toFixed(1) + 'h capacity">';
        html += '<span class="pl-tl-job-ref">' + esc(jobLabel) + '</span>';
        html += '<span class="pl-tl-job-hours">' + seg.hours.toFixed(1) + 'h</span>';
        html += '</div>';
      });

      // Show available capacity indicator
      if (segmentsForDay.length === 0 && availableHours > 0) {
        html += '<div class="pl-tl-available" title="Available: ' + availableHours.toFixed(1) + 'h"></div>';
      }

      html += '</div>';
    }

    html += '</div>';
    return html;
  }

  /**
   * Render a single machine row in True Gantt mode
   * Jobs are positioned based on date_start/date_end, not capacity fill
   * @param {Object} machine Machine data
   * @param {Array} planningDays Planning days array
   * @param {number} dayWidth Day column width in pixels
   */
  function renderMachineRowGantt(machine, planningDays, dayWidth) {
    var html = '<div class="pl-tl-row pl-tl-machine-row pl-tl-gantt-row" data-machine-id="' + machine.id + '">';

    // Machine label cell (sticky)
    html += '<div class="pl-tl-machine-cell">';
    html += '<div class="pl-tl-machine-ref">' + esc(machine.ref) + '</div>';
    if (machine.label && machine.label !== machine.ref) {
      html += '<div class="pl-tl-machine-label">' + esc(machine.label) + '</div>';
    }
    html += '</div>';

    // Calculate view range from planningDays
    if (planningDays.length === 0) {
      html += '</div>';
      return html;
    }
    var viewStartDate = planningDays[0].date;
    var viewEndDate = planningDays[planningDays.length - 1].date;
    var viewStartTs = Date.parse(viewStartDate);
    var viewEndTs = Date.parse(viewEndDate) + 86400000; // Add 1 day
    var viewRangeMs = viewEndTs - viewStartTs;

    // Container for gantt bars (overlaid on day cells)
    html += '<div class="pl-tl-gantt-bars-container">';

    // Render each job as a single bar positioned by date range
    machine.jobs.forEach(function(job) {
      // Only render jobs with date_start
      if (!job.date_start) return;

      // Parse job dates
      var jobStartTs = Date.parse(job.date_start);
      // Priority: date_end (manual) > computed_end (auto from qty/rate) > duration fallback
      var jobEndTs;
      if (job.date_end) {
        jobEndTs = Date.parse(job.date_end) + 86400000;
      } else if (job.computed_end) {
        jobEndTs = job.computed_end * 1000; // Convert from seconds to milliseconds
      } else {
        // Fallback: use duration_hours (from estimated_hours or qty_per_hr)
        var durationHours = job.duration_hours || job.remaining_hours_used || 1;
        jobEndTs = jobStartTs + (durationHours * 3600000);
      }

      // Skip if job is completely outside view range
      if (jobEndTs <= viewStartTs || jobStartTs >= viewEndTs) return;

      // Clamp job range to view range
      var barStartTs = Math.max(jobStartTs, viewStartTs);
      var barEndTs = Math.min(jobEndTs, viewEndTs);

      // Calculate left and width as percentage of view
      var leftPercent = ((barStartTs - viewStartTs) / viewRangeMs) * 100;
      var widthPercent = ((barEndTs - barStartTs) / viewRangeMs) * 100;

      var jobLabel = job.works_order_no || job.fg || ('Job#' + job.id);
      html += '<div class="pl-tl-job-gantt-bar" ';
      html += 'data-machine-id="' + machine.id + '" ';
      html += 'data-job-id="' + job.id + '" ';
      html += 'style="left: ' + leftPercent.toFixed(1) + '%; width: ' + widthPercent.toFixed(1) + '%;" ';
      html += 'title="' + esc(jobLabel) + '">';
      html += '<span class="pl-tl-gantt-bar-label">' + esc(jobLabel) + '</span>';
      html += '</div>';
    });

    html += '</div>'; // End gantt-bars-container

    // Day cells (visual reference grid, clickable for adding jobs in gantt mode)
    for (var dayIdx = 0; dayIdx < planningDays.length; dayIdx++) {
      var dayDate = planningDays[dayIdx].date;
      html += '<div class="pl-tl-day-cell pl-tl-gantt-day-cell" data-day="' + dayIdx + '" data-date="' + dayDate + '" data-machine-id="' + machine.id + '"></div>';
    }

    html += '</div>';
    return html;
  }

  /**
   * Select a job and show details in the card panel
   */
  function selectJob(machineId, jobId) {
    machineId = parseInt(machineId, 10);
    jobId = parseInt(jobId, 10);

    // Find machine and job
    var machine = null;
    var job = null;
    if (timelineData && timelineData.machines) {
      timelineData.machines.forEach(function(m) {
        if (m.id === machineId) {
          machine = m;
          m.jobs.forEach(function(j) {
            if (j.id === jobId) job = j;
          });
        }
      });
    }

    if (!job) return;

    selectedJob = { machine: machine, job: job };

    // Highlight selected block
    document.querySelectorAll('.pl-tl-job-block').forEach(function(b) {
      b.classList.remove('pl-tl-job-selected');
    });
    document.querySelectorAll('.pl-tl-job-block[data-job-id="' + jobId + '"]').forEach(function(b) {
      b.classList.add('pl-tl-job-selected');
    });

    // Render job card
    renderJobCard(machine, job);
  }

  /**
   * Render job details card (view mode or edit mode)
   */
  function renderJobCard(machine, job) {
    var card = $('plJobCard');
    if (!card) return;

    if (editMode) {
      renderJobCardEditMode(machine, job);
    } else {
      renderJobCardViewMode(machine, job);
    }
  }

  /**
   * Render job card in view mode (read-only)
   */
  function renderJobCardViewMode(machine, job) {
    var card = $('plJobCard');
    if (!card) return;

    var jobTitle = job.works_order_no || job.fg || ('Job#' + job.id);
    var html = '<div class="pl-job-card-header">';
    html += '<div class="pl-job-card-title">' + esc(jobTitle) + '</div>';
    html += '<div class="pl-job-card-header-actions">';
    if (config.canEditjob) {
      html += '<button type="button" class="pl-btn-edit-job" id="plBtnEditJob">Edit</button>';
    }
    if (config.canDeletejob) {
      html += '<button type="button" class="pl-btn-delete-job" id="plBtnDeleteJob">Delete</button>';
    }
    html += '</div>';
    html += '</div>';
    html += '<div class="pl-job-card-machine">' + esc(machine.ref) + '</div>';

    html += '<div class="pl-job-card-body">';

    // Key info row - hours summary
    html += '<div class="pl-job-card-row">';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">' + (job.remaining_hours_used || 0).toFixed(1) + 'h</span>';
    html += '<span class="pl-job-card-stat-label">Remaining</span>';
    html += '</div>';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">' + (job.estimated_hours || 0).toFixed(1) + 'h</span>';
    html += '<span class="pl-job-card-stat-label">Estimated</span>';
    html += '</div>';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">#' + (job.sort_order || 0) + '</span>';
    html += '<span class="pl-job-card-stat-label">Order</span>';
    html += '</div>';
    html += '</div>';

    // Override notice
    if (job.remaining_hours_override !== null && job.remaining_hours_override !== '') {
      html += '<div class="pl-job-card-notice">Hours override active</div>';
    }

    // Date range if set
    if (job.date_start || job.date_end) {
      html += '<div class="pl-job-card-section">';
      html += '<div class="pl-job-card-label">Scheduled Time</div>';
      html += '<div class="pl-job-card-dates">';
      if (job.date_start) {
        html += '<div class="pl-job-card-date"><span>Start:</span> ' + esc(formatDatetime(job.date_start)) + '</div>';
      }
      if (job.date_end) {
        html += '<div class="pl-job-card-date"><span>End:</span> ' + esc(formatDatetime(job.date_end)) + '</div>';
      }
      html += '</div>';
      html += '</div>';
    }

    // Schedule segments
    html += '<div class="pl-job-card-section">';
    html += '<div class="pl-job-card-label">Schedule</div>';
    html += '<div class="pl-job-card-segments">';
    job.segments.forEach(function(seg) {
      if (seg.overflow) {
        html += '<div class="pl-job-card-seg pl-job-card-seg-overflow">Overflow: ' + seg.hours.toFixed(1) + 'h</div>';
      } else {
        var dayLabel = seg.day_index === 0 ? 'Today' : 'D+' + seg.day_index;
        html += '<div class="pl-job-card-seg">' + dayLabel + ': ' + seg.hours.toFixed(1) + 'h</div>';
      }
    });
    html += '</div>';
    html += '</div>';

    // Notes if present
    if (job.notes) {
      html += '<div class="pl-job-card-section">';
      html += '<div class="pl-job-card-label">Notes</div>';
      html += '<div class="pl-job-card-notes">' + esc(job.notes) + '</div>';
      html += '</div>';
    }

    // Collapsible details
    html += '<details class="pl-job-card-details">';
    html += '<summary>More details</summary>';
    html += '<div class="pl-job-card-details-body">';
    if (job.fg) {
      html += '<div class="pl-job-card-detail"><span>FG:</span> ' + esc(job.fg) + '</div>';
    }
    if (job.works_order_no) {
      html += '<div class="pl-job-card-detail"><span>Works Order:</span> ' + esc(job.works_order_no) + '</div>';
    }
    html += '<div class="pl-job-card-detail"><span>Status:</span> ' + esc(job.status || 'planned') + '</div>';
    html += '<div class="pl-job-card-detail"><span>Group:</span> ' + esc(job.group_code || config.group) + '</div>';
    html += '<div class="pl-job-card-detail"><span>Job ID:</span> ' + job.id + '</div>';
    html += '</div>';
    html += '</details>';

    html += '</div>'; // body

    card.innerHTML = html;

    // Bind edit button click
    var btnEdit = $('plBtnEditJob');
    if (btnEdit) {
      btnEdit.addEventListener('click', function() {
        enterEditMode();
      });
    }

    // Bind delete button click
    var btnDelete = $('plBtnDeleteJob');
    if (btnDelete) {
      btnDelete.addEventListener('click', function() {
        confirmDeleteJob(job.id);
      });
    }
  }

  /**
   * Render job card in edit mode (editable form)
   */
  function renderJobCardEditMode(machine, job) {
    var card = $('plJobCard');
    if (!card) return;

    var jobTitle = job.works_order_no || job.fg || ('Job#' + job.id);
    var html = '<div class="pl-job-card-header pl-job-card-edit-header">';
    html += '<div class="pl-job-card-title">Editing: ' + esc(jobTitle) + '</div>';
    html += '</div>';

    html += '<form id="plEditJobForm" class="pl-job-card-edit-form">';
    html += '<div class="pl-job-card-body">';

    // Workstation (read-only) - format: "Workstation: Forming FM7" or "Workstation: Trimming TM3"
    var groupCode = job.group_code || config.group;
    var lblTxt = (machine.label || '').trim();
    var refTxt = (machine.ref || '').trim();
    var machineLabel = (lblTxt && refTxt && lblTxt !== refTxt) ? (lblTxt + ' - ' + refTxt) : (lblTxt || refTxt);
    var wsDisplayText = 'Workstation: ' + esc(machineLabel);
    html += '<div class="pl-form-group">';
    html += '<div class="pl-form-readonly" id="plEditJobWorkstationDisplay">' + wsDisplayText + '</div>';
    html += '<input type="hidden" name="fk_workstation" value="' + machine.id + '">';
    html += '<input type="hidden" name="group_code" value="' + esc(groupCode) + '">';
    html += '</div>';

    // Product (FG) - Autocomplete (like Add Job)
    html += '<div class="pl-form-group">';
    html += '<label class="pl-form-label">Product (FG)</label>';
    html += '<div class="pl-autocomplete-container">';
    html += '<input type="text" class="pl-form-input pl-edit-product-search" placeholder="Search by ref or name..." data-job-id="' + job.id + '">';
    html += '<ul class="pl-autocomplete-results pl-edit-product-results" data-job-id="' + job.id + '"></ul>';
    html += '</div>';
    html += '<input type="hidden" name="fk_product" class="pl-edit-product-hidden" data-job-id="' + job.id + '" value="' + (job.fk_product || '') + '">';
    html += '</div>';

    // Status select
    html += '<div class="pl-form-group">';
    html += '<label class="pl-form-label">Status</label>';
    html += '<select name="status" class="pl-form-input">';
    var statuses = ['planned', 'in_progress', 'paused', 'completed', 'cancelled'];
    var currentStatus = job.status || 'planned';
    statuses.forEach(function(st) {
      var sel = st === currentStatus ? ' selected' : '';
      html += '<option value="' + st + '"' + sel + '>' + esc(st) + '</option>';
    });
    html += '</select>';
    html += '</div>';

    // Quantity
    html += '<div class="pl-form-group">';
    html += '<label class="pl-form-label">Quantity</label>';
    html += '<input type="number" name="qty" class="pl-form-input" step="0.01" min="0.01" value="' + (job.qty || 1.00) + '">';
    html += '</div>';

    // Date start - default to Today at 06:30 if missing
    html += '<div class="pl-form-group">';
    html += '<label class="pl-form-label">Start</label>';
    var startVal = '';
    if (job.date_start) {
      startVal = formatDatetimeForInput(job.date_start);
    } else {
      // Default: Today at 06:30
      var defaultStart = new Date();
      defaultStart.setHours(6, 30, 0, 0);
      startVal = formatDatetimeLocal(defaultStart);
    }
    html += '<input type="datetime-local" name="date_start" class="pl-form-input" value="' + startVal + '">';
    html += '</div>';

    // Notes
    html += '<div class="pl-form-group">';
    html += '<label class="pl-form-label">Notes</label>';
    html += '<textarea name="notes" class="pl-form-input" rows="2">' + esc(job.notes || '') + '</textarea>';
    html += '</div>';

    // Error display
    html += '<div class="pl-form-error" id="plEditJobError"></div>';

    html += '</div>'; // body

    // Footer with Save/Cancel buttons
    html += '<div class="pl-job-card-edit-footer">';
    html += '<button type="button" class="pl-btn pl-btn-cancel" id="plEditJobCancel">Cancel</button>';
    html += '<button type="submit" class="pl-btn pl-btn-submit" id="plEditJobSave">Save</button>';
    html += '</div>';

    html += '</form>';

    card.innerHTML = html;

    // Bind form events
    var form = $('plEditJobForm');
    var btnCancel = $('plEditJobCancel');

    if (form) {
      form.addEventListener('submit', function(e) {
        e.preventDefault();
        submitEditJob(job.id);
      });
    }
    if (btnCancel) {
      btnCancel.addEventListener('click', function() {
        cancelEditMode();
      });
    }

    // TG2.1: Prefill product in edit mode (set hidden fk_product and display text)
    // Always fetch fk_product from backend if not in job data
    var hiddenField = form ? form.querySelector('input[name="fk_product"]') : null;
    var searchInput = form ? form.querySelector('.pl-edit-product-search') : null;
    
    // If job already has fk_product (from cache), use it directly
    if (job.fk_product && job.fk_product > 0) {
      fetchAndDisplayProduct(job.fk_product, hiddenField, searchInput);
    } 
    // Otherwise fetch job details to get fk_product
    else if (job.id) {
      loadJobDetailsForEdit(job.id, function(fkProduct) {
        if (fkProduct && fkProduct > 0) {
          fetchAndDisplayProduct(fkProduct, hiddenField, searchInput);
        } else {
          // No product found
          if (searchInput) searchInput.value = '';
        }
      });
    }
    // Fallback: clear product field
    else {
      if (searchInput) searchInput.value = '';
    }

    // TG2.1: Bind onChange handlers for qty, product, group to recalc end time (client-side preview only)
    var qtyInputEdit = form ? form.querySelector('input[name="qty"]') : null;
    var productSearchEdit = form ? form.querySelector('.pl-edit-product-search') : null;
    var productHiddenEdit = form ? form.querySelector('input[name="fk_product"]') : null;
    var groupSelectEdit = form ? form.querySelector('select[name="group_code"]') : null;
    var startInputEdit = form ? form.querySelector('input[name="date_start"]') : null;

    // Bind product autocomplete for edit mode
    if (productSearchEdit) {
      // Flag to prevent clearing during initial prefill
      // Set to true if prefill will happen (i.e., job has ID and we'll fetch product)
      var isInitialized = !!(job.fk_product && job.fk_product > 0) || !!job.id;
      
      productSearchEdit.addEventListener('input', function() {
        var query = this.value.trim();
        
        // Only check for clearing if user interaction (not prefill)
        if (query.length < 2) {
          var resultsList = form ? form.querySelector('.pl-edit-product-results') : null;
          if (resultsList) resultsList.innerHTML = '';
          
          // Only clear fk_product if user explicitly cleared the search
          if (productHiddenEdit && !query && isInitialized) {
            productHiddenEdit.value = '';
          }
          return;
        }
        
        // User is searching
        isInitialized = true;
        searchProductsForEdit(query, form);
      });
    }

    // Note: estimated_hours and date_end fields removed (TG2.1 - server-side calculation only)
  }

  /**
   * Format datetime from database format for display
   * @param {string} dt Database datetime string (YYYY-MM-DD HH:MM:SS)
   * @returns {string} Formatted display string
   */
  function formatDatetime(dt) {
    if (!dt) return '';
    // Replace T if present, ensure consistent format
    dt = dt.replace('T', ' ');
    var parts = dt.split(' ');
    if (parts.length < 2) return dt;
    var datePart = parts[0].split('-');
    var timePart = parts[1].substring(0, 5); // HH:MM
    if (datePart.length < 3) return dt;
    return datePart[2] + '/' + datePart[1] + ' ' + timePart;
  }

  /**
   * Format datetime for input[type=datetime-local] from database format
   * @param {string} dt Database datetime string (YYYY-MM-DD HH:MM:SS)
   * @returns {string} YYYY-MM-DDTHH:MM format
   */
  function formatDatetimeForInput(dt) {
    if (!dt) return '';
    // Handle both formats
    dt = dt.replace('T', ' ');
    var parts = dt.split(' ');
    if (parts.length < 2) return '';
    var datePart = parts[0];
    var timePart = parts[1].substring(0, 5); // HH:MM
    return datePart + 'T' + timePart;
  }

  /**
   * Enter edit mode for the selected job
   */
  function enterEditMode() {
    if (!selectedJob) return;
    console.log('[EDIT] open', { job_id: selectedJob.job.id });
    editMode = true;
    // Store original data for cancel
    editOriginalData = JSON.parse(JSON.stringify(selectedJob.job));
    renderJobCard(selectedJob.machine, selectedJob.job);
  }

  /**
   * Cancel edit mode and revert to view mode
   */
  function cancelEditMode() {
    editMode = false;
    if (selectedJob && editOriginalData) {
      selectedJob.job = editOriginalData;
    }
    editOriginalData = null;
    if (selectedJob) {
      renderJobCard(selectedJob.machine, selectedJob.job);
    }
  }

  /**
   * Submit job edits via AJAX
   * @param {number} jobId Job ID
   */
  function submitEditJob(jobId) {
    var form = $('plEditJobForm');
    var errorDiv = $('plEditJobError');
    var saveBtn = $('plEditJobSave');
    if (!form || !config.updatejobEndpoint) return;

    // Clear error
    if (errorDiv) errorDiv.textContent = '';

    // Check for "Loading..." state or invalid product
    var productSearchInput = form.querySelector('.pl-edit-product-search');
    if (productSearchInput && productSearchInput.value === 'Loading...') {
      if (errorDiv) errorDiv.textContent = 'Product is still loading. Please wait...';
      return;
    }

    // Validate fk_product (required)
    var fkProductField = form.querySelector('input[name="fk_product"]');
    if (!fkProductField || !fkProductField.value || parseInt(fkProductField.value, 10) <= 0) {
      if (errorDiv) errorDiv.textContent = 'Please select a product (FG).';
      return;
    }

    // Gather form data
    var formData = new FormData(form);
    // Remove group_code and fk_workstation - these are read-only and not editable
    formData.delete('group_code');
    formData.delete('fk_workstation');
    formData.append('token', config.token);
    formData.append('job_id', jobId);

    // Disable save button
    if (saveBtn) {
      saveBtn.disabled = true;
      saveBtn.textContent = 'Saving...';
    }

    fetch(config.updatejobEndpoint, {
      method: 'POST',
      credentials: 'same-origin',
      body: formData
    })
    .then(function(r) {
      if (!r.ok) {
        return r.text().then(function(text) {
          throw new Error('Server error: ' + r.status + ' - ' + (text ? text.substring(0, 100) : 'no response'));
        });
      }
      return r.json().catch(function(err) {
        throw new Error('Invalid JSON response: ' + err.message);
      });
    })
    .then(function(data) {
      if (data.error) {
        if (errorDiv) errorDiv.textContent = data.error;
        return;
      }
      if (!data.success) {
        if (errorDiv) errorDiv.textContent = 'Unknown error occurred';
        return;
      }
      // Success - exit edit mode and refresh timeline
      editMode = false;
      editOriginalData = null;
      selectedJob = null;
      // Show placeholder in card
      var card = $('plJobCard');
      if (card) {
        card.innerHTML = '<div class="pl-job-card-placeholder">Select a job to view details</div>';
      }
      // Reset fetch state and force immediate timeline refresh (TG2.1: date_end now calculated server-side)
      isFetching = false;
      pendingFetch = false;
      fetchTimeline();
    })
    .catch(function(err) {
      if (errorDiv) errorDiv.textContent = 'Request failed: ' + err.message;
    })
    .finally(function() {
      if (saveBtn) {
        saveBtn.disabled = false;
        saveBtn.textContent = 'Save';
      }
    });
  }

  /**
   * Confirm and delete a job
   * @param {number} jobId Job ID
   */
  function confirmDeleteJob(jobId) {
    if (!selectedJob) return;
    var jobTitle = selectedJob.job.works_order_no || selectedJob.job.fg || ('Job#' + jobId);
    var msg = 'Delete Job ' + esc(jobTitle) + '?\n\nThis action cannot be undone.';
    if (confirm(msg)) {
      deleteJob(jobId);
    }
  }

  /**
   * Submit job deletion via AJAX
   * @param {number} jobId Job ID
   */
  function deleteJob(jobId) {
    if (!config.deletejobEndpoint) return;

    var formData = new FormData();
    formData.append('token', config.token);
    formData.append('job_id', jobId);

    fetch(config.deletejobEndpoint, {
      method: 'POST',
      credentials: 'same-origin',
      body: formData
    })
    .then(function(r) {
      if (!r.ok) {
        return r.text().then(function(text) {
          throw new Error('Server error: ' + r.status + ' - ' + (text ? text.substring(0, 100) : 'no response'));
        });
      }
      return r.json().catch(function(err) {
        throw new Error('Invalid JSON response: ' + err.message);
      });
    })
    .then(function(data) {
      if (data.error) {
        alert('Error: ' + data.error);
        return;
      }
      if (!data.success) {
        alert('Unknown error occurred');
        return;
      }
      // Success - clear panel and refresh timeline
      selectedJob = null;
      var card = $('plJobCard');
      if (card) {
        card.innerHTML = '<div class="pl-job-card-placeholder">Select a job to view details</div>';
      }
      fetchTimeline();
    })
    .catch(function(err) {
      alert('Request failed: ' + err.message);
    });
  }

  /**
   * Toggle past/today view
   */
  window.plTimelineTogglePast = function() {
    showPastMode = !showPastMode;
    var btn = $('plBtnPastToggle');
    if (btn) {
      btn.textContent = showPastMode ? 'Now' : 'Past';
      btn.classList.toggle('pl-btn-past-active', showPastMode);
    }
    fetchTimeline();
  };

  /**
   * Change days parameter
   */
  window.plTimelineChangeDays = function(days) {
    var url = new URL(window.location.href);
    url.searchParams.set('days', days);
    window.location.href = url.toString();
  };

  // ========== ADD JOB MODAL ==========

  /**
   * Format datetime for input[type=datetime-local]
   * @param {Date} date
   * @returns {string} YYYY-MM-DDTHH:MM
   */
  function formatDatetimeLocal(date) {
    var y = date.getFullYear();
    var m = String(date.getMonth() + 1).padStart(2, '0');
    var d = String(date.getDate()).padStart(2, '0');
    var hh = String(date.getHours()).padStart(2, '0');
    var mm = String(date.getMinutes()).padStart(2, '0');
    return y + '-' + m + '-' + d + 'T' + hh + ':' + mm;
  }

  /**
   * Update end time based on TG2.1 model: qty + product rate, fallback to estimated hours
   * Priority: (qty / qty_per_hour_*) > estimated_hours
   */
  function updateEndTime() {
    // Removed: end time is now calculated server-side based on qty + rate
    // This function is kept as stub for compatibility but is not called in Add Job flow
    return;
  }

  /**
   * Product autocomplete - debounce state
   */
  var productAutocompleteState = {
    debounceTimer: null,
    abortController: null
  };

  /**
   * Search products via AJAX autocomplete
   */
  function searchProducts(query) {
    var resultsList = $('plAddJobProductResults');
    if (!resultsList) return;

    // Cancel previous request
    if (productAutocompleteState.abortController) {
      productAutocompleteState.abortController.abort();
    }

    // Clear if query too short
    if (!query || query.length < 2) {
      resultsList.innerHTML = '';
      return;
    }

    // Create new abort controller
    productAutocompleteState.abortController = new AbortController();

    fetch(config.searchProductsEndpoint + '?q=' + encodeURIComponent(query), {
      method: 'GET',
      credentials: 'same-origin',
      signal: productAutocompleteState.abortController.signal
    })
    .then(function(r) {
      if (!r.ok) {
        throw new Error('Server error: ' + r.status);
      }
      return r.json();
    })
    .then(function(data) {
      if (!data.success || !data.products) {
        resultsList.innerHTML = '<li class="pl-autocomplete-no-results">No results</li>';
        return;
      }

      if (data.products.length === 0) {
        resultsList.innerHTML = '<li class="pl-autocomplete-no-results">No results</li>';
        return;
      }

      var html = '';
      data.products.forEach(function(prd) {
        var displayText = prd.ref + (prd.label && prd.label !== prd.ref ? ' - ' + prd.label : '');
        html += '<li class="pl-autocomplete-item" data-id="' + prd.id + '" data-ref="' + esc(prd.ref) + '" data-label="' + esc(displayText) + '">';
        html += esc(displayText);
        html += '</li>';
      });
      resultsList.innerHTML = html;

      // Bind click handlers
      resultsList.querySelectorAll('.pl-autocomplete-item').forEach(function(item) {
        item.addEventListener('click', function() {
          selectProduct(this.dataset.id, this.dataset.label);
        });
      });
    })
    .catch(function(err) {
      if (err.name !== 'AbortError') {
        if (debugEnabled) {
          console.error('Product search failed:', err);
        }
        resultsList.innerHTML = '<li class="pl-autocomplete-no-results">Error loading results</li>';
      }
    });
  }

  /**
   * Select product from autocomplete
   */
  function selectProduct(productId, displayText) {
    var searchInput = $('plAddJobProductSearch');
    var hiddenField = $('plAddJobProduct');
    var resultsList = $('plAddJobProductResults');

    if (searchInput) searchInput.value = displayText;
    if (hiddenField) hiddenField.value = parseInt(productId, 10);
    if (resultsList) resultsList.innerHTML = '';
  }

  /**
   * Fetch product details (ref, label) by ID from backend search endpoint
   * Used to prefill edit form when product_ref/label not in cached job data
   */
  function fetchProductDetails(productId, callback) {
    if (!config.searchProductsEndpoint || !productId) {
      if (callback) callback(null);
      return;
    }

    fetch(config.searchProductsEndpoint + '?id=' + encodeURIComponent(productId), {
      method: 'GET',
      credentials: 'same-origin'
    })
    .then(function(r) {
      if (!r.ok) throw new Error('Server error: ' + r.status);
      return r.json();
    })
    .then(function(data) {
      if (data.success && data.products && data.products.length > 0) {
        if (callback) callback(data.products[0]);
      } else {
        if (callback) callback(null);
      }
    })
    .catch(function(err) {
      if (debugEnabled) console.error('Product fetch failed:', err);
      if (callback) callback(null);
    });
  }

  /**
   * Fetch product details for Edit mode (direct ref element update)
   * Fetches product ref and label, then updates search input directly
   */


  /**
   * Load job details from backend (to get fk_product if not in event data)
   * Minimal endpoint: ajax/get_job.php?id=... returns {success, fk_product, ...}
   */
  function loadJobDetailsForEdit(jobId, callback) {
    // Try config endpoint first (if exists), fallback to minimal get_job endpoint
    var endpoint = null;
    if (config.updatejobEndpoint) {
      // Can read job via update endpoint with GET, or use dedicated read endpoint
      endpoint = config.updatejobEndpoint.replace('update_job', 'get_job');
    }
    
    if (!endpoint) {
      // Fallback: try standard pattern
      var baseUrl = window.location.href.split('?')[0].split('#')[0];
      var ajaxDir = baseUrl.substring(0, baseUrl.lastIndexOf('/')) + '/ajax/';
      endpoint = ajaxDir + 'get_job.php';
    }
    
    if (!endpoint || !jobId) {
      if (callback) callback(null);
      return;
    }

    console.log('[EDIT] fetch job/product', { job_id: jobId, url: endpoint });
    fetch(endpoint + '?id=' + encodeURIComponent(jobId), {
      method: 'GET',
      credentials: 'same-origin'
    })
    .then(function(r) {
      if (!r.ok) throw new Error('Server error: ' + r.status);
      return r.json();
    })
    .then(function(data) {
      console.log('[EDIT] ajax success', data);
      if (data.success && data.fk_product) {
        if (callback) callback(data.fk_product);
      } else {
        if (callback) callback(null);
      }
    })
    .catch(function(err) {
      console.error('[EDIT] ajax error', err);
      if (debugEnabled) console.error('Job details fetch failed:', err);
      if (callback) callback(null);
    });
  }

  /**
   * Fetch product by ID and display in search input
   * Handles sequence: fetch product → extract ref/label → update input
   */
  function fetchAndDisplayProduct(productId, hiddenField, searchInput) {
    if (!productId || productId <= 0 || !searchInput) {
      if (searchInput) searchInput.value = '';
      return;
    }

    // Set hidden field
    if (hiddenField) {
      hiddenField.value = productId;
    }

    // Show loading state
    if (searchInput) {
      searchInput.value = 'Loading...';
    }

    // Try to find in config cache first
    if (config.products && config.products.length > 0) {
      var product = config.products.find(function(p) { return p.id === productId; });
      if (product && product.ref) {
        var displayText = product.ref;
        if (product.label && product.label !== product.ref) {
          displayText += ' - ' + product.label;
        }
        console.log('[EDIT] set product', { fk_product: productId, displayText: displayText });
        if (searchInput) searchInput.value = displayText;
        return;
      }
    }

    // Fetch from get_product.php (by ID lookup, not search_products which is autocomplete)
    var baseUrl = window.location.href.split('?')[0].split('#')[0];
    var ajaxDir = baseUrl.substring(0, baseUrl.lastIndexOf('/')) + '/ajax/';
    var endpoint = ajaxDir + 'get_product.php';
    console.log('[EDIT] fetch job/product', { product_id: productId, url: endpoint });
    fetchProductDetailsForEditDirect(endpoint, productId, searchInput);
  }

  /**
   * Direct fetch for product details using custom endpoint
   */
  function fetchProductDetailsForEditDirect(endpoint, productId, searchInput) {
    if (!endpoint || !productId || !searchInput) {
      return;
    }

    fetch(endpoint + '?id=' + encodeURIComponent(productId), {
      method: 'GET',
      credentials: 'same-origin'
    })
    .then(function(r) {
      if (!r.ok) throw new Error('Server error: ' + r.status);
      return r.json();
    })
    .then(function(data) {
      console.log('[EDIT] ajax success', data);
      if (data.success && data.ref) {
        var displayText = data.ref;
        if (data.label && data.label !== data.ref) {
          displayText += ' - ' + data.label;
        }
        console.log('[EDIT] set product', { fk_product: productId, displayText: displayText });
        searchInput.value = displayText;
      } else {
        searchInput.value = '';
      }
    })
    .catch(function(err) {
      console.error('[EDIT] ajax error', err);
      if (debugEnabled) console.error('Product details fetch failed:', err);
      searchInput.value = '';
    });
  }

  /**
   * Search products for Edit mode (uses class selectors to find form context)
   */
  function searchProductsForEdit(query, form) {
    var resultsList = form ? form.querySelector('.pl-edit-product-results') : null;
    if (!resultsList) return;

    if (!query || query.length < 2) {
      resultsList.innerHTML = '';
      return;
    }

    fetch(config.searchProductsEndpoint + '?q=' + encodeURIComponent(query), {
      method: 'GET',
      credentials: 'same-origin'
    })
    .then(function(r) {
      if (!r.ok) throw new Error('Server error: ' + r.status);
      return r.json();
    })
    .then(function(data) {
      if (!data.success || !data.products || data.products.length === 0) {
        resultsList.innerHTML = '<li class="pl-autocomplete-no-results">No results</li>';
        return;
      }

      var html = '';
      data.products.forEach(function(prd) {
        var displayText = prd.ref + (prd.label && prd.label !== prd.ref ? ' - ' + prd.label : '');
        html += '<li class="pl-autocomplete-item" data-id="' + prd.id + '" data-ref="' + esc(prd.ref) + '" data-label="' + esc(displayText) + '">';
        html += esc(displayText);
        html += '</li>';
      });
      resultsList.innerHTML = html;

      // Bind click handlers for edit mode
      resultsList.querySelectorAll('.pl-autocomplete-item').forEach(function(item) {
        item.addEventListener('click', function() {
          selectProductForEdit(form, this.dataset.id, this.dataset.label);
        });
      });
    })
    .catch(function(err) {
      if (debugEnabled) console.error('Product search failed:', err);
      resultsList.innerHTML = '<li class="pl-autocomplete-no-results">Error loading results</li>';
    });
  }

  /**
   * Select product for Edit mode (update hidden field and search input in form context)
   */
  function selectProductForEdit(form, productId, displayText) {
    if (!form) return;
    
    var searchInput = form.querySelector('.pl-edit-product-search');
    var hiddenField = form.querySelector('input[name="fk_product"]');
    var resultsList = form.querySelector('.pl-edit-product-results');

    if (searchInput) searchInput.value = displayText;
    if (hiddenField) hiddenField.value = parseInt(productId, 10);
    if (resultsList) resultsList.innerHTML = '';
  }

  /**
   * Get group code from workstation label or metadata
   * If workstation has group data, use it; else infer from label (Forming→forming, Trimming→trimming)
   * @param {number} fkWorkstation Workstation ID
   * @param {string} workstationLabel Workstation label
   * @returns {string} 'forming' or 'trimming'
   */
  function getGroupCodeFromWorkstation(fkWorkstation, workstationLabel) {
    if (!workstationLabel) {
      // Fallback to current tab if no label provided
      return getActiveGroupCode();
    }
    
    // Try to infer from label: "Forming FM7" → forming, "Trimming TM3" → trimming
    var labelLower = workstationLabel.toLowerCase();
    if (labelLower.indexOf('forming') >= 0) return 'forming';
    if (labelLower.indexOf('trimming') >= 0) return 'trimming';
    
    // Fallback
    return getActiveGroupCode();
  }

  /**
   * Get active group code from tabs or current config
   * Looks for active tab element and extracts group code (forming/trimming)
   * @returns {string} 'forming' or 'trimming'
   */
  function getActiveGroupCode() {
    // Try to find active tab
    var activeTab = document.querySelector('.pl-tab-active');
    if (activeTab) {
      var href = activeTab.getAttribute('href');
      if (href && href.indexOf('group=forming') >= 0) return 'forming';
      if (href && href.indexOf('group=trimming') >= 0) return 'trimming';
    }
    // Fallback to config
    return config.group || 'forming';
  }

  /**
   * Get workstation context from clicked cell/row element
   * @param {HTMLElement} el Clicked element (cell, row, or parent)
   * @returns {object} {fk_workstation: number, workstation_label: string}
   */
  function getCellContext(el) {
    // Look up DOM tree for data attributes
    var current = el;
    var maxDepth = 5;
    while (current && maxDepth-- > 0) {
      var wsId = current.dataset.workstationId || current.dataset.fkWorkstation;
      var wsLabel = current.dataset.workstationLabel || current.dataset.workstationName;
      
      if (wsId) {
        return {
          fk_workstation: parseInt(wsId, 10),
          workstation_label: wsLabel || 'Workstation #' + wsId
        };
      }
      
      // Try to read label from row header if available
      if (current.classList && current.classList.contains('pl-tl-row')) {
        // Try to read label/ref from row header
        var labelEl = current.querySelector('.pl-tl-machine-label');
        var refEl = current.querySelector('.pl-tl-machine-ref');
        var lblTxt = (labelEl && labelEl.textContent) ? labelEl.textContent.trim() : '';
        var refTxt = (refEl && refEl.textContent) ? refEl.textContent.trim() : '';
        wsLabel = (lblTxt && refTxt && lblTxt !== refTxt) ? (lblTxt + ' - ' + refTxt) : (lblTxt || refTxt);
        wsId = current.dataset.workstationId || current.dataset.machineId;
        if (wsId) {
          return {
            fk_workstation: parseInt(wsId, 10),
            workstation_label: wsLabel || ('Workstation #' + wsId)
          };
        }
      }
      
      current = current.parentElement;
    }
    
    // Fallback: if no context found
    return {
      fk_workstation: 0,
      workstation_label: ''
    };
  }


  /**
   * Open the Add Job modal
   */
  function openAddJobModal() {
    var modal = $('plAddJobModal');
    if (!modal) return;
    modal.classList.add('pl-modal-open');
    
    // Reset form
    var form = $('plAddJobForm');
    if (form) form.reset();
    
    // Clear product autocomplete
    var searchInput = $('plAddJobProductSearch');
    var resultsList = $('plAddJobProductResults');
    var hiddenField = $('plAddJobProduct');
    if (searchInput) searchInput.value = '';
    if (hiddenField) hiddenField.value = '';
    if (resultsList) resultsList.innerHTML = '';
    
    // Workstation display will be set by openAddJobModalWithPrefill if called with prefill
    var wsDisplay = $('plAddJobWorkstationDisplay');
    if (wsDisplay) wsDisplay.textContent = '-';
    var wsHidden = $('plAddJobWorkstationHidden');
    if (wsHidden) wsHidden.value = '';
    var groupHidden = $('plAddJobGroupHidden');
    if (groupHidden) groupHidden.value = '';
    
    // Clear start
    var startInput = $('plAddJobStart');
    if (startInput) startInput.value = '';
    
    // Clear error
    var errorDiv = $('plAddJobError');
    if (errorDiv) errorDiv.textContent = '';
  }

  /**
   * Open the Add Job modal with prefilled values from cell click
   * @param {string} machineId Workstation ID
   * @param {string} machineLabel Workstation label (optional)
   * @param {string} date Date string YYYY-MM-DD
   */
  function openAddJobModalWithPrefill(machineId, machineLabel, date) {
    openAddJobModal();

    // Calculate group_code from workstation label
    var groupCode = getGroupCodeFromWorkstation(machineId, machineLabel);

    // Prefill workstation display and hidden fields
    var wsDisplay = $('plAddJobWorkstationDisplay');
    var wsHidden = $('plAddJobWorkstationHidden');
    var groupHidden = $('plAddJobGroupHidden');
    
    if (wsDisplay) {
      var displayText = 'Workstation: ' + (machineLabel || ('Machine #' + machineId));
      wsDisplay.textContent = displayText;
    }
    if (wsHidden) {
      wsHidden.value = machineId;
    }
    if (groupHidden) {
      groupHidden.value = groupCode;
    }

    // Prefill start: date at 06:30 (planning day boundary)
    var startInput = $('plAddJobStart');
    if (startInput && date) {
      var startDate = new Date(date + 'T06:30:00');
      startInput.value = formatDatetimeLocal(startDate);
    }
  }

  /**
   * Close the Add Job modal
   */
  function closeAddJobModal() {
    var modal = $('plAddJobModal');
    if (!modal) return;
    modal.classList.remove('pl-modal-open');
  }

  /**
   * Validate Add Job form fields before submission
   * Returns error message if validation fails, or null if valid
   */
  function validateAddJobForm() {
    var fkWorkstation = document.querySelector('input[name="fk_workstation"]');
    var fkProduct = document.querySelector('input[name="fk_product"]');
    var qty = document.querySelector('input[name="qty"]');
    var dateStart = document.querySelector('input[name="date_start"]');

    // Validate fk_workstation (required, from hidden field)
    if (!fkWorkstation || !fkWorkstation.value || parseInt(fkWorkstation.value, 10) <= 0) {
      return 'Please select a workstation from timeline.';
    }

    // Validate fk_product (required)
    if (!fkProduct || !fkProduct.value || parseInt(fkProduct.value, 10) <= 0) {
      return 'Please select a product (FG).';
    }

    // Validate qty (required, must be > 0)
    if (!qty || !qty.value) {
      return 'Please enter a quantity.';
    }
    var qtyVal = parseFloat(qty.value);
    if (isNaN(qtyVal) || qtyVal <= 0) {
      return 'Quantity must be greater than 0.';
    }

    // Validate date_start (required)
    if (!dateStart || !dateStart.value) {
      return 'Please select a start date and time.';
    }

    // All validation passed
    return null;
  }

  /**
   * Submit the Add Job form
   * Validates fields before sending, prevents double-submit, handles timeout
   */
  function submitAddJobForm(e) {
    e.preventDefault();

    var form = $('plAddJobForm');
    var errorDiv = $('plAddJobError');
    var submitBtn = $('plModalSubmit');
    if (!form || !config.addjobEndpoint) return;

    // Clear error
    if (errorDiv) errorDiv.textContent = '';

    // Prevent double-submit
    if (submitAddJobForm.isSubmitting) return;
    submitAddJobForm.isSubmitting = true;

    // Frontend validation BEFORE fetch
    var validationError = validateAddJobForm();
    if (validationError) {
      if (errorDiv) errorDiv.textContent = validationError;
      submitAddJobForm.isSubmitting = false;
      return;
    }

    // Gather form data
    var formData = new FormData(form);
    formData.append('token', config.token);

    // Disable submit button
    if (submitBtn) {
      submitBtn.disabled = true;
      submitBtn.textContent = 'Adding...';
    }

    // Setup timeout (15 seconds)
    var abortController = new AbortController();
    var timeoutId = setTimeout(function() {
      abortController.abort();
    }, 15000);

    try {
      fetch(config.addjobEndpoint, {
        method: 'POST',
        credentials: 'same-origin',
        body: formData,
        signal: abortController.signal
      })
      .then(function(r) {
        clearTimeout(timeoutId);
        if (!r.ok) {
          return r.text().then(function(text) {
            throw new Error('Server error: ' + r.status + ' - ' + (text ? text.substring(0, 100) : 'no response'));
          });
        }
        return r.json().catch(function(err) {
          throw new Error('Invalid JSON response: ' + err.message);
        });
      })
      .then(function(data) {
        if (data.error) {
          var errorMsg = data.error;
          // Append debug info if available
          if (data.debug_message || data.debug_file) {
            errorMsg += '\n\nDebug Info:\n';
            if (data.debug_message) errorMsg += '  Message: ' + data.debug_message + '\n';
            if (data.debug_file) errorMsg += '  File: ' + data.debug_file + '\n';
            if (data.debug_line) errorMsg += '  Line: ' + data.debug_line + '\n';
            if (data.db_error) errorMsg += '  DB Error: ' + data.db_error + '\n';
          }
          if (errorDiv) errorDiv.textContent = errorMsg;
          return;
        }
        if (!data.success) {
          if (errorDiv) errorDiv.textContent = 'Unknown error occurred';
          return;
        }
        // Success - close modal and refresh timeline
        closeAddJobModal();
        fetchTimeline();
      })
      .catch(function(err) {
        clearTimeout(timeoutId);
        if (err.name === 'AbortError') {
          if (errorDiv) errorDiv.textContent = 'Request timed out (server not responding). Please try again.';
        } else {
          if (errorDiv) errorDiv.textContent = 'Request failed: ' + err.message;
        }
      })
      .finally(function() {
        clearTimeout(timeoutId);
        submitAddJobForm.isSubmitting = false;
        if (submitBtn) {
          submitBtn.disabled = false;
          submitBtn.textContent = 'Add Job';
        }
      });
    } catch (err) {
      clearTimeout(timeoutId);
      submitAddJobForm.isSubmitting = false;
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = 'Add Job';
      }
      if (errorDiv) errorDiv.textContent = 'Request failed: ' + err.message;
    }
  }

  /**
   * Bind Add Job modal events
   */
  function bindAddJobEvents() {
    // Skip if already bound (safety against duplicate handlers)
    if (window.__planningAddJobBound) {
      return;
    }
    window.__planningAddJobBound = true;

    // Bind Past toggle button
    var btnPastToggle = $('plBtnPastToggle');
    if (btnPastToggle) {
      btnPastToggle.addEventListener('click', plTimelineTogglePast);
    }

    if (!config.canAddjob) return;

    var btnClose = $('plModalClose');
    var btnCancel = $('plModalCancel');
    var modal = $('plAddJobModal');
    var form = $('plAddJobForm');

    if (btnClose) {
      btnClose.addEventListener('click', closeAddJobModal);
    }
    if (btnCancel) {
      btnCancel.addEventListener('click', closeAddJobModal);
    }
    if (modal) {
      modal.addEventListener('click', function(e) {
        if (e.target === modal) closeAddJobModal();
      });
    }
    if (form) {
      form.addEventListener('submit', submitAddJobForm);
    }
    
    // Product autocomplete input listener
    var productSearch = $('plAddJobProductSearch');
    if (productSearch) {
      productSearch.addEventListener('input', function(e) {
        var query = e.target.value.trim();
        
        // Clear debounce timer
        if (productAutocompleteState.debounceTimer) {
          clearTimeout(productAutocompleteState.debounceTimer);
        }
        
        // Debounce search (250ms)
        productAutocompleteState.debounceTimer = setTimeout(function() {
          searchProducts(query);
        }, 250);
      });

      // Click outside results list to close
      var resultsList = $('plAddJobProductResults');
      if (resultsList) {
        document.addEventListener('click', function(e) {
          if (!e.target.closest('.pl-autocomplete-container')) {
            resultsList.innerHTML = '';
          }
        });
      }
    }

    // ESC key to close modal or autocomplete
    document.addEventListener('keydown', function(e) {
      if (e.key === 'Escape') {
        var resultsList = $('plAddJobProductResults');
        // First close autocomplete if it's open
        if (resultsList && resultsList.innerHTML.trim()) {
          resultsList.innerHTML = '';
          e.stopPropagation();
          return;
        }
        // Then close modal
        if (modal && modal.classList.contains('pl-modal-open')) {
          closeAddJobModal();
        }
      }
    });

    // End time is now calculated server-side based on qty + product rate
    // No client-side calculation needed for Add Job modal
  }

  /**
   * Initialize on DOM ready (bind-once guard to prevent duplicate handlers)
   */
  function init() {
    // Prevent rebinding on refresh/re-render
    if (window.__planningBound) {
      return;
    }
    window.__planningBound = true;

    if (!loadConfig()) {
      renderError('Timeline configuration missing.');
      return;
    }
    fetchTimeline();
    bindAddJobEvents();
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
