# POST /custom/planning/ajax/add_job.php Debug Guide

## Summary of Changes

Added proper debug info support for POST requests to `add_job.php`. Debug details are **only included when debug mode is enabled** (security-safe).

### Changes Made

#### 1. [ajax/add_job.php](ajax/add_job.php) - PHP Backend

**Key modifications:**

- **Exception Handler** (lines 57-97): 
  - Changed error response from generic "Server error processing request" to actual `$ex->getMessage()`
  - Added conditional debug fields for POST requests when `isDebugMode()` returns true:
    - `debug_message`
    - `debug_file`
    - `debug_line`
    - `db_error` (if database error exists)
  - All errors logged to `dol_syslog(..., LOG_ERR)` regardless of debug mode

- **Shutdown Handler** (lines 99-135):
  - Same improvements for fatal PHP errors
  - Extracts file, line, and database error info
  - Conditionally includes debug fields for POST requests when debug enabled

- **Main Try-Catch Block** (lines 381-407):
  - Enhanced error response with proper message and debug details
  - Logs full error context with `dol_syslog`
  - Returns separate fields (`debug_message`, `debug_file`, `debug_line`, `db_error`) instead of concatenated string

#### 2. [js/timeline.js](js/timeline.js) - JavaScript Frontend

**Key modification:**

- **Form submission handler** (lines 1150-1169):
  - Updated error display to check for debug fields in response
  - If debug info is present, appends it to error message for developer visibility:
    ```
    Debug Info:
      Message: ...
      File: ...
      Line: ...
      DB Error: ...
    ```

---

## How to Enable Debug for POST Calls

### Option 1: Add `?debug=1` to Endpoint URL (Recommended for Modal)

Modify [timeline.php](timeline.php) at the point where `$addJobEndpoint` is set:

**Current:**
```php
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php';
```

**To enable debug:**
```php
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php?debug=1';
```

### Option 2: Use Global PLANNING_DEBUG Config

In your Dolibarr configuration, set:
```php
$conf->global->PLANNING_DEBUG = 1;
```

This enables debug for **all** planning endpoints globally.

### Option 3: Admin User

Any admin user automatically has debug enabled (checked in `isDebugMode()` function).

---

## Testing

### Before Fix (No Debug)
**POST request with error:**
```json
{
  "success": false,
  "error": "Server error processing request"
}
```
→ No useful information about what went wrong.

### After Fix (With ?debug=1)
**POST request with error (modal will show):**
```json
{
  "error": "Product rate (qty_per_hour_form) is not configured or is invalid...",
  "debug_message": "Product rate (qty_per_hour_form) is not configured...",
  "debug_file": "/var/www/dolibarr/custom/planning/ajax/add_job.php",
  "debug_line": 313,
  "db_error": null
}
```

**Server logs (always):**
```
Planning Add Job Error: Product rate ... in /var/www/dolibarr/custom/planning/ajax/add_job.php:313
```

---

## Security Notes

- ✅ Debug fields are **only** included when:
  - Request is `POST` (not GET), AND
  - Debug mode is enabled (`?debug=1` OR `PLANNING_DEBUG=1` OR user is admin)
  
- ✅ CSRF token validation still required (unchanged)

- ✅ Debug is disabled for GET requests (even if `?debug=1` is passed)

- ✅ No database changes, no UI framework changes

---

## JSON Response Schema (on error with debug enabled)

```json
{
  "error": "string - human readable error message (always)",
  "debug_message": "string - exception message (debug only)",
  "debug_file": "string - file path (debug only)",
  "debug_line": "number - line number (debug only)",
  "db_error": "string - database error if any (debug only)"
}
```

---

## Minimal Diff Summary

| File | Changes | Lines |
|------|---------|-------|
| `ajax/add_job.php` | Error handlers now conditionally include debug fields for POST requests | 57-97, 99-135, 381-407 |
| `js/timeline.js` | Form error display now shows debug info when present | 1150-1169 |

Total modifications: **~40 lines** across 2 files (minimal, surgical changes).
