# Before & After Comparison

## Current Situation (Before)

### Direct GET: `GET /ajax/add_job.php?debug=1`
```
Response: "Invalid or missing security token."
Status: 403 ✓ (expected)
```

### Modal POST: `POST /ajax/add_job.php`
When an error occurs in background:
```
Modal shows:   "Request failed: Server error: 500 - [no response]"
Browser logs:  Empty (no JSON response)
Server logs:   Error details (but developer must SSH to check)
Debugging:     ❌ Very difficult - no file/line info in response
```

**Problem:** Developer gets generic 500 error, must check server logs manually.

---

## After Implementation

### Direct GET: `GET /ajax/add_job.php?debug=1`
```
Response: "Invalid or missing security token."
Status: 403 ✓ (unchanged - GET requests never show debug)
```

### Modal POST: `POST /ajax/add_job.php?debug=1`
When an error occurs:
```json
Response: {
  "error": "Product rate (qty_per_hour_form) is not configured...",
  "debug_message": "Product rate (qty_per_hour_form) is not configured...",
  "debug_file": "/custom/planning/ajax/add_job.php",
  "debug_line": 313,
  "db_error": null
}
Status: 500
```

**Modal displays:**
```
Product rate (qty_per_hour_form) is not configured or is invalid. 
Please set the production rate for this product.

Debug Info:
  Message: Product rate (qty_per_hour_form) is not configured...
  File: /custom/planning/ajax/add_job.php
  Line: 313
  DB Error: 
```

**Server logs (always, even without debug):**
```
Planning Add Job Error: Product rate (qty_per_hour_form) is not configured... 
  in /custom/planning/ajax/add_job.php:313
```

**Debugging:** ✅ Complete - developer sees exact file, line, and error immediately.

---

## Production Mode (Without Debug)

### Modal POST: `POST /ajax/add_job.php` (no ?debug=1)
```json
Response: {
  "error": "Product rate (qty_per_hour_form) is not configured..."
}
Status: 500
```

**Modal displays:**
```
Product rate (qty_per_hour_form) is not configured or is invalid. 
Please set the production rate for this product.
```

**Server logs (still detailed):**
```
Planning Add Job Error: Product rate (qty_per_hour_form) is not configured... 
  in /custom/planning/ajax/add_job.php:313
```

**Result:** ✅ Users see friendly error, developers can check server logs if needed.

---

## Comparison Table

| Aspect | Before | After (Debug Off) | After (Debug On) |
|--------|--------|------------------|------------------|
| User sees friendly error | ❌ No | ✅ Yes | ✅ Yes |
| Modal shows file/line | ❌ No | ❌ No | ✅ Yes |
| Modal shows DB error | ❌ No | ❌ No | ✅ Yes (if any) |
| Server logs detailed error | ✅ Yes | ✅ Yes | ✅ Yes |
| GET requests show debug | N/A | N/A | ❌ No (secure) |
| CSRF token required | ✅ Yes | ✅ Yes | ✅ Yes |
| Production safe | ✅ Yes | ✅ Yes | ❌ (debug mode only) |

---

## Code Changes

### Backend (PHP)
```diff
// Exception handler - BEFORE
$response = ['error' => 'Server error processing request'];
if (isDebugMode()) {
    $response['debug'] = $ex->getMessage() . ' @ ' . $ex->getFile() . ':' . $ex->getLine();
}

// Exception handler - AFTER
$response = ['error' => $ex->getMessage()];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode()) {
    $response['debug_message'] = $ex->getMessage();
    $response['debug_file'] = $ex->getFile();
    $response['debug_line'] = $ex->getLine();
    if (!empty($dbErr)) {
        $response['db_error'] = $dbErr;
    }
}
```

### Frontend (JavaScript)
```diff
// Form error handler - BEFORE
if (data.error) {
    if (errorDiv) errorDiv.textContent = data.error;
    return;
}

// Form error handler - AFTER
if (data.error) {
    var errorMsg = data.error;
    if (data.debug_message || data.debug_file) {
        errorMsg += '\n\nDebug Info:\n';
        if (data.debug_message) errorMsg += '  Message: ' + data.debug_message + '\n';
        if (data.debug_file) errorMsg += '  File: ' + data.debug_file + '\n';
        if (data.debug_line) errorMsg += '  Line: ' + data.debug_line + '\n';
        if (data.db_error) errorMsg += '  DB Error: ' + data.db_error + '\n';
    }
    if (errorDiv) errorDiv.textContent = errorMsg;
    return;
}
```

---

## Real-World Scenarios

### Scenario 1: Product Missing Rate Configuration

**Before:**
```
❌ User: "Why can't I add a job?"
✓ Modal: "Server error: 500"
❌ Developer: Must SSH to server, grep logs, find line 313, check what's there
⏱️ Time to debug: 5-10 minutes
```

**After (with debug):**
```
❌ User: "I see an error"
✓ Modal: "Product rate (qty_per_hour_form) is not configured...
          File: /custom/planning/ajax/add_job.php
          Line: 313"
✓ Developer: Immediately knows it's a product configuration issue
⏱️ Time to debug: 30 seconds
```

### Scenario 2: Database Connection Lost

**Before:**
```
Modal: "Server error: 500"
Developer: SSHes in, checks logs, sees "Lost connection to MySQL"
Action: Restart database
```

**After (with debug):**
```
Modal: "Database error: Lost connection to MySQL server
         DB Error: Lost connection to MySQL server during query"
Developer: Immediately knows it's a DB connection issue
Action: Restart database (same, but faster identification)
```

---

## Implementation Status

- [x] Exception handler returns structured debug fields (POST only)
- [x] Shutdown handler returns structured debug fields (POST only)
- [x] Main catch block includes debug fields (POST only)
- [x] All errors logged to dol_syslog (always)
- [x] JavaScript displays debug fields if present
- [x] CSRF token validation unchanged
- [x] GET requests never show debug (security)
- [x] Debug is opt-in (not default)
- [x] No database changes
- [x] No UI framework changes
- [x] Backward compatible

---

## How to Enable

**For Testing:** Add to [timeline.php](timeline.php):
```php
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php?debug=1';
```

**For Production:** Set in config:
```php
$conf->global->PLANNING_DEBUG = 1;
```

**For Admins:** Automatic (no change needed).

---

## Summary

✅ **Problem Solved:** POST requests to add_job.php now return useful debug info when enabled.  
✅ **Secure:** Debug only shows for POST + enabled, never for GET.  
✅ **Minimal Impact:** ~40 lines across 2 files.  
✅ **Production Ready:** No breaking changes, backward compatible.  
✅ **Quick Enable:** Add `?debug=1` to endpoint URL.
