<?php
/**
 * Planning Timeline - JSON data endpoint
 * Returns timeline model for Forming/Trimming groups.
 *
 * Parameters:
 *   group  - 'forming' or 'trimming'
 *   start  - start planning day (YYYY-MM-DD)
 *   days   - number of days (1-60, default 14)
 */

// ===== GLOBAL ERROR HANDLERS - Must be at top =====
// Prevent direct output before headers
ob_start();

/**
 * Check if debug mode should be enabled
 * Debug mode is ON if: ?debug=1, PLANNING_DEBUG global, or user is admin
 */
function isDebugMode() {
    global $user;
    if (GETPOST('debug', 'int') == 1) {
        return true;
    }
    if (!empty($GLOBALS['conf']->global->PLANNING_DEBUG)) {
        return true;
    }
    if (!empty($user) && !empty($user->admin)) {
        return true;
    }
    return false;
}

// Global error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    global $db;
    $msg = "PHP Error [$errno] in $errfile:$errline - $errstr";
    dol_syslog('Planning Timeline Data: ' . $msg, LOG_ERR);
    return false;
});

// Global exception handler
set_exception_handler(function($ex) {
    global $db, $user;
    $msg = "Exception: " . $ex->getMessage() . " in " . $ex->getFile() . ":" . $ex->getLine();
    dol_syslog('Planning Timeline Data: ' . $msg, LOG_ERR);
    
    // Log stack trace
    if (!empty($ex->getTraceAsString())) {
        dol_syslog('Planning Timeline Data: Stack trace: ' . $ex->getTraceAsString(), LOG_ERR);
    }
    
    // Log DB error if applicable
    if (!empty($db)) {
        $dbErr = $db->lasterror();
        if (!empty($dbErr)) {
            dol_syslog('Planning Timeline Data: DB error: ' . $dbErr, LOG_ERR);
        }
    }
    
    ob_clean();
    http_response_code(500);
    
    $response = [
        'error' => 'Server error processing request'
    ];
    
    // Include debug info if enabled
    if (isDebugMode()) {
        $response['debug'] = $ex->getMessage() . ' @ ' . $ex->getFile() . ':' . $ex->getLine();
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                $response['db_error'] = $dbErr;
            }
        }
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
});

// Shutdown handler - catch any fatal errors
register_shutdown_function(function() {
    global $db, $user;
    $error = error_get_last();
    if ($error !== null && ($error['type'] & (E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR))) {
        $msg = "Fatal Error [" . $error['type'] . "] in " . $error['file'] . ":" . $error['line'] . " - " . $error['message'];
        dol_syslog('Planning Timeline Data: ' . $msg, LOG_ERR);
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                dol_syslog('Planning Timeline Data: DB error: ' . $dbErr, LOG_ERR);
            }
        }
        ob_clean();
        http_response_code(500);
        
        $response = [
            'error' => 'Server error processing request'
        ];
        
        // Include debug info if enabled
        if (isDebugMode()) {
            $response['debug'] = $error['message'] . ' @ ' . $error['file'] . ':' . $error['line'];
            if (!empty($db)) {
                $dbErr = $db->lasterror();
                if (!empty($dbErr)) {
                    $response['db_error'] = $dbErr;
                }
            }
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    }
});

require '../../../main.inc.php';
require_once __DIR__ . '/../class/PlanningTimelineService.class.php';

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// Clear any buffered output
ob_end_clean();

/**
 * Output JSON response and exit.
 *
 * @param array $data Response data
 * @param int   $code HTTP status code
 */
function outputJson($data, $code = 200)
{
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

// Check permissions
if (empty($user->rights->planning) || empty($user->rights->planning->read)) {
    outputJson(array('error' => 'Access denied. Missing planning read permission.'), 403);
}

// Get parameters
$group = GETPOST('group', 'aZ09');
$start = GETPOST('start', 'alphanohtml');
$days = GETPOST('days', 'int');

// Validate group
if (empty($group)) {
    outputJson(array('error' => 'Missing required parameter: group'), 400);
}

// Default start to today if not provided
if (empty($start)) {
    $start = date('Y-m-d');
}

// Default days to 14 if not provided
if (empty($days) || $days < 1) {
    $days = 14;
}
if ($days > 60) {
    $days = 60;
}

// Compute timeline
try {
    $service = new PlanningTimelineService($db);
    $result = $service->computeTimeline($group, $start, $days);

    if (isset($result['error'])) {
        outputJson($result, 400);
    }

    // Debug output (only when debug=1 or PLANNING_DEBUG enabled)
    if (GETPOST('debug', 'int') == 1 || !empty($conf->global->PLANNING_DEBUG)) {
        $result['debug_conf_entity'] = $conf->entity;
        $result['debug_jobs_sql'] = isset($service->debugJobsInfo['sql']) ? $service->debugJobsInfo['sql'] : null;
        $result['debug_jobs_count'] = isset($service->debugJobsInfo['count']) ? $service->debugJobsInfo['count'] : null;
        $result['debug_machineIds'] = isset($service->debugJobsInfo['machineIds']) ? $service->debugJobsInfo['machineIds'] : null;
    }

    outputJson($result, 200);
} catch (Exception $e) {
    outputJson(array('error' => 'Internal error: ' . $e->getMessage()), 500);
}