<?php
/**
 * Planning - Add Job AJAX endpoint
 * Creates a new planning job with automatic end date calculation.
 *
 * Required POST parameters:
 *   token          - Dolibarr CSRF token
 *   group_code     - 'forming' or 'trimming'
 *   fk_workstation - workstation ID
 *   fk_product     - product ID (required for rate lookup)
 *   qty            - quantity (must be > 0)
 *   date_start     - start datetime (YYYY-MM-DDTHH:MM or YYYY-MM-DD HH:MM:SS)
 *
 * Optional POST parameters:
 *   works_order_no - works order number
 *   notes          - notes text
 *
 * Notes:
 * - estimated_hours and date_end parameters are IGNORED (calculated server-side)
 * - End date is automatically calculated: start + (qty / rate) + buffer
 * - rate comes from product extrafields (qty_per_hour_form or qty_per_hour_trim)
 * - If rate is missing/<=0, returns error
 */

// ===== GLOBAL ERROR HANDLERS - Must be at top =====
// Prevent direct output before headers
ob_start();

/**
 * Check if debug mode should be enabled
 * Debug mode is ON if: ?debug=1, PLANNING_DEBUG global, or user is admin
 */
function isDebugMode() {
    global $user;
    if (GETPOST('debug', 'int') == 1) {
        return true;
    }
    if (!empty($GLOBALS['conf']->global->PLANNING_DEBUG)) {
        return true;
    }
    if (!empty($user) && !empty($user->admin)) {
        return true;
    }
    return false;
}

// Global error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    global $db, $user;
    $msg = "PHP Error [$errno] in $errfile:$errline - $errstr";
    dol_syslog('Planning Add Job: ' . $msg, LOG_ERR);
    // Return false to allow default handler
    return false;
});

// Global exception handler
set_exception_handler(function($ex) {
    global $db, $user;
    $msg = "Exception: " . $ex->getMessage() . " in " . $ex->getFile() . ":" . $ex->getLine();
    dol_syslog('Planning Add Job: ' . $msg, LOG_ERR);
    
    // Log stack trace
    if (!empty($ex->getTraceAsString())) {
        dol_syslog('Planning Add Job: Stack trace: ' . $ex->getTraceAsString(), LOG_ERR);
    }
    
    // Log DB error if applicable
    $dbErr = '';
    if (!empty($db)) {
        $dbErr = $db->lasterror();
        if (!empty($dbErr)) {
            dol_syslog('Planning Add Job: DB error: ' . $dbErr, LOG_ERR);
        }
    }
    
    ob_clean();
    http_response_code(500);
    
    $response = [
        'success' => false,
        'error' => $ex->getMessage()
    ];
    
    // Include debug info if enabled and POST request
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode()) {
        $response['debug_message'] = $ex->getMessage();
        $response['debug_file'] = $ex->getFile();
        $response['debug_line'] = $ex->getLine();
        if (!empty($dbErr)) {
            $response['db_error'] = $dbErr;
        }
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
});

// Shutdown handler - catch any fatal errors
register_shutdown_function(function() {
    global $db, $user;
    $error = error_get_last();
    if ($error !== null && ($error['type'] & (E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR))) {
        $msg = "Fatal Error [" . $error['type'] . "] in " . $error['file'] . ":" . $error['line'] . " - " . $error['message'];
        dol_syslog('Planning Add Job: ' . $msg, LOG_ERR);
        
        $dbErr = '';
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                dol_syslog('Planning Add Job: DB error: ' . $dbErr, LOG_ERR);
            }
        }
        
        ob_clean();
        http_response_code(500);
        
        $response = [
            'success' => false,
            'error' => $error['message']
        ];
        
        // Include debug info if enabled and POST request
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode()) {
            $response['debug_message'] = $error['message'];
            $response['debug_file'] = $error['file'];
            $response['debug_line'] = $error['line'];
            if (!empty($dbErr)) {
                $response['db_error'] = $dbErr;
            }
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    }
});

require '../../../main.inc.php';

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// Clear any buffered output
ob_end_clean();

/**
 * Output JSON response and exit.
 *
 * @param array $data Response data
 * @param int   $code HTTP status code
 */
function outputJson($data, $code = 200)
{
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Load production rate from product extrafields
 * Uses Dolibarr Product class with array_options (extrafields with options_ prefix)
 *
 * @param DoliDB $db Database handler
 * @param int    $fk_product Product ID
 * @param string $groupCode 'forming' or 'trimming'
 * @return float|null Rate in pieces/hour, or null if not found/invalid
 */
function loadProductRate($db, $fk_product, $groupCode)
{
    if (!$fk_product) {
        return null;
    }

    try {
        require_once(DOL_DOCUMENT_ROOT . '/product/class/product.class.php');
        require_once(DOL_DOCUMENT_ROOT . '/core/class/extrafields.class.php');

        $product = new Product($db);
        $res = $product->fetch((int)$fk_product);
        if ($res <= 0) {
            dol_syslog('Planning: Product fetch failed for ID ' . $fk_product, LOG_WARNING);
            return null;
        }

        // Load extrafields into array_options
        $extrafields = new ExtraFields($db);
        $extralabels = $extrafields->fetch_name_optionals_label($product->table_element, true);
        if (count($extralabels) > 0) {
            $product->fetch_optionals($product->id, $extralabels);
        }

        // Get rate from array_options with options_ prefix
        $fieldname = ($groupCode === 'forming') ? 'options_qty_per_hour_form' : 'options_qty_per_hour_trim';

        $arrayOpts = $product->array_options ?? array();
        if (isset($arrayOpts[$fieldname]) && !empty($arrayOpts[$fieldname])) {
            $rate = (float)$arrayOpts[$fieldname];
            dol_syslog('Planning: Product #' . $fk_product . ' rate (' . $fieldname . ') = ' . $rate, LOG_DEBUG);
            return $rate > 0 ? $rate : null;
        }

        // Debug: log available keys if rate not found
        dol_syslog('Planning: Product #' . $fk_product . ' missing rate field ' . $fieldname . '. Available options: ' . implode(', ', array_keys($arrayOpts)), LOG_DEBUG);

        return null;
    } catch (Exception $e) {
        dol_syslog('Planning: Error loading product rate: ' . $e->getMessage(), LOG_ERR);
        return null;
    }
}

/**
 * Get buffer minutes based on group code
 *
 * @param string $groupCode 'forming' or 'trimming'
 * @return int Buffer in minutes
 */
function getBufferForGroupCode($groupCode)
{
    if ($groupCode === 'forming') {
        return 60; // 1 hour for forming
    } elseif ($groupCode === 'trimming') {
        return 30; // 30 minutes for trimming
    }
    return 60; // Default fallback
}

// Check CSRF token
$token = GETPOST('token', 'aZ09');
$sessionToken = isset($_SESSION['newtoken']) ? $_SESSION['newtoken'] : (isset($_SESSION['token']) ? $_SESSION['token'] : '');
if (empty($token) || $token !== $sessionToken) {
    outputJson(array('error' => 'Invalid or missing security token.'), 403);
}

// Check permissions - require addjob right
if (empty($user->rights->planning) || empty($user->rights->planning->addjob)) {
    outputJson(array('error' => 'Access denied. Missing planning addjob permission.'), 403);
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    outputJson(array('error' => 'Invalid request method.'), 405);
}

// Get and validate parameters (early, fast validation)
$group_code = GETPOST('group_code', 'aZ09');
$fk_workstation = GETPOST('fk_workstation', 'int');
$fk_product = GETPOST('fk_product', 'int');
$works_order_no = GETPOST('works_order_no', 'alphanohtml');
$qty = GETPOST('qty', 'alpha');
$date_start = GETPOST('date_start', 'alphanohtml');
$notes = GETPOST('notes', 'restricthtml');

// FAST validation: required fields (no DB queries yet)
if (empty($group_code) || !in_array($group_code, array('forming', 'trimming'))) {
    outputJson(array('error' => 'Invalid group code. Must be forming or trimming.'), 400);
}

if (empty($fk_workstation) || $fk_workstation <= 0) {
    outputJson(array('error' => 'Please select a workstation.'), 400);
}

if (empty($fk_product) || $fk_product <= 0) {
    outputJson(array('error' => 'Product (FG) is required.'), 400);
}

// Parse and validate qty (before any DB queries)
$qty = str_replace(',', '.', $qty);
$qty = (float) $qty;
if ($qty <= 0) {
    outputJson(array('error' => 'Quantity must be greater than 0.'), 400);
}

// Parse and format start datetime (before any DB queries)
$date_start_sql = null;
if (!empty($date_start)) {
    $date_start = str_replace('T', ' ', $date_start);
    if (strlen($date_start) === 16) {
        $date_start .= ':00'; // Add seconds
    }
    $ts = strtotime($date_start);
    if ($ts !== false) {
        $date_start_sql = date('Y-m-d H:i:s', $ts);
    } else {
        outputJson(array('error' => 'Invalid start date format.'), 400);
    }
} else {
    outputJson(array('error' => 'Start date is required.'), 400);
}

// SLOW validation: check existence in DB (after fast validation passes)
$sqlWsCheck = "SELECT rowid FROM " . MAIN_DB_PREFIX . "workstation_workstation WHERE rowid = " . (int) $fk_workstation;
$resWsCheck = $db->query($sqlWsCheck);
if (!$resWsCheck || !$db->fetch_object($resWsCheck)) {
    outputJson(array('error' => 'Selected workstation does not exist.'), 400);
}

$sqlPrdCheck = "SELECT rowid FROM " . MAIN_DB_PREFIX . "product WHERE rowid = " . (int) $fk_product;
$resPrdCheck = $db->query($sqlPrdCheck);
if (!$resPrdCheck || !$db->fetch_object($resPrdCheck)) {
    outputJson(array('error' => 'Selected product does not exist.'), 400);
}

// Begin transaction
$db->begin();

try {
    // Load product rate and calculate end date
    $rate = loadProductRate($db, $fk_product, $group_code);
    if ($rate === null || $rate <= 0) {
        throw new Exception('Product rate (' . ($group_code === 'forming' ? 'qty_per_hour_form' : 'qty_per_hour_trim') . ') is not configured or is invalid. Please set the production rate for this product.');
    }

    // Calculate end date = start + (qty / rate) + buffer
    $durationHours = $qty / $rate;
    $bufferMinutes = getBufferForGroupCode($group_code);
    $durationSeconds = ($durationHours * 3600) + ($bufferMinutes * 60);

    $startTs = strtotime($date_start_sql);
    $endTs = $startTs + $durationSeconds;
    $date_end_sql = date('Y-m-d H:i:s', $endTs);

    // Get next sort_order for this workstation + group combination
    $sqlMax = "SELECT COALESCE(MAX(sort_order), 0) + 10 as next_sort
               FROM " . MAIN_DB_PREFIX . "planning_job
               WHERE fk_workstation = " . (int) $fk_workstation . "
               AND group_code = '" . $db->escape($group_code) . "'";
    $resMax = $db->query($sqlMax);
    $nextSort = 10;
    if ($resMax && ($objMax = $db->fetch_object($resMax))) {
        $nextSort = (int) $objMax->next_sort;
    }

    // Insert job
    $sql = "INSERT INTO " . MAIN_DB_PREFIX . "planning_job (
                fk_workstation,
                fk_product,
                qty,
                works_order_no,
                notes,
                group_code,
                status,
                sort_order,
                date_start,
                date_end,
                entity,
                fk_user_creat,
                tms
            ) VALUES (
                " . (int) $fk_workstation . ",
                " . (int) $fk_product . ",
                " . (float) $qty . ",
                '" . $db->escape($works_order_no) . "',
                '" . $db->escape($notes) . "',
                '" . $db->escape($group_code) . "',
                'planned',
                " . (int) $nextSort . ",
                '" . $date_start_sql . "',
                '" . $date_end_sql . "',
                " . (int) $conf->entity . ",
                " . (int) $user->id . ",
                NOW()
            )";

    $result = $db->query($sql);
    if (!$result) {
        throw new Exception('Database error: ' . $db->lasterror());
    }

    $newId = $db->last_insert_id(MAIN_DB_PREFIX . 'planning_job');

    $db->commit();

    outputJson(array(
        'success' => true,
        'message' => 'Job created successfully.',
        'job_id' => (int) $newId
    ), 200);

} catch (Exception $e) {
    $db->rollback();
    
    $errorMsg = $e->getMessage();
    $errorFile = $e->getFile();
    $errorLine = $e->getLine();
    
    // Log full error details
    dol_syslog('Planning Add Job Error: ' . $errorMsg . ' in ' . $errorFile . ':' . $errorLine, LOG_ERR);
    $dbErr = !empty($db) ? $db->lasterror() : '';
    if (!empty($dbErr)) {
        dol_syslog('Planning Add Job DB Error: ' . $dbErr, LOG_ERR);
    }
    
    $response = array('error' => $errorMsg);
    
    // Include debug details if debug mode is enabled
    if (isDebugMode()) {
        $response['debug_message'] = $errorMsg;
        $response['debug_file'] = $errorFile;
        $response['debug_line'] = $errorLine;
        if (!empty($dbErr)) {
            $response['db_error'] = $dbErr;
        }
    }
    
    outputJson($response, 500);
}