<?php
/**
 * Planning Timeline Service
 * Computes timeline model for Planning Timeline view.
 * Uses planning day boundary 06:30 -> 06:30 (next calendar day).
 */

class PlanningTimelineService
{
    /** @var DoliDB */
    private $db;

    /** Planning day starts at 06:30 */
    const DAY_START_HOUR = 6;
    const DAY_START_MINUTE = 30;

    /**
     * @param DoliDB $db Database handler
     */
    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Compute timeline model for a group of machines.
     *
     * @param string $groupCode   Group code: 'forming' or 'trimming'
     * @param string $startDay    Start planning day (YYYY-MM-DD)
     * @param int    $daysLimit   Number of planning days to compute
     * @return array              Timeline model or error array
     */
    public function computeTimeline($groupCode, $startDay, $daysLimit)
    {
        $groupCode = strtolower(trim($groupCode));
        if (!in_array($groupCode, array('forming', 'trimming'))) {
            return array('error' => 'Invalid group_code. Must be forming or trimming.');
        }

        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDay)) {
            return array('error' => 'Invalid start_day format. Use YYYY-MM-DD.');
        }

        $daysLimit = (int) $daysLimit;
        if ($daysLimit < 1 || $daysLimit > 60) {
            return array('error' => 'days_limit must be between 1 and 60.');
        }

        // Compute planning day boundaries
        $planningDays = $this->computePlanningDays($startDay, $daysLimit);

        // Load machines for group
        $machines = $this->loadMachines($groupCode);
        if (isset($machines['error'])) {
            return $machines;
        }

        // Load capacity for machines and days
        $machineIds = array_keys($machines);
        $capacities = $this->loadCapacities($machineIds, $startDay, $daysLimit);
        $overrides = $this->loadCapacityOverrides($machineIds, $startDay, $daysLimit);
        $downtimes = $this->loadDowntimes($machineIds, $planningDays);

        // Load jobs for machines
        $jobs = $this->loadJobs($machineIds, $groupCode);

        // Build timeline model
        $result = array(
            'group_code' => $groupCode,
            'start_day' => $startDay,
            'days_limit' => $daysLimit,
            'planning_days' => $planningDays,
            'machines' => array(),
        );

        foreach ($machines as $machineId => $machine) {
            $machineModel = array(
                'id' => $machine['id'],
                'ref' => $machine['ref'],
                'label' => $machine['label'],
                'days' => array(),
                'jobs' => array(),
            );

            // Build day capacity info
            // Default 24h/day if no capacity configured
            $defaultCapacity = 24;
            for ($dayIdx = 0; $dayIdx < $daysLimit; $dayIdx++) {
                $dayDate = $planningDays[$dayIdx]['date'];
                $baseCapacity = isset($capacities[$machineId]) ? (float) $capacities[$machineId] : $defaultCapacity;
                $override = isset($overrides[$machineId][$dayDate]) ? $overrides[$machineId][$dayDate] : null;
                $effectiveCapacity = ($override !== null) ? (float) $override : $baseCapacity;

                // Subtract downtime
                $downtimeHours = isset($downtimes[$machineId][$dayDate]) ? (float) $downtimes[$machineId][$dayDate] : 0;
                $availableHours = max(0, $effectiveCapacity - $downtimeHours);

                $machineModel['days'][$dayIdx] = array(
                    'date' => $dayDate,
                    'base_capacity' => $baseCapacity,
                    'override' => $override,
                    'effective_capacity' => $effectiveCapacity,
                    'downtime_hours' => $downtimeHours,
                    'available_hours' => $availableHours,
                );
            }

            // Build job segments
            $machineJobs = isset($jobs[$machineId]) ? $jobs[$machineId] : array();
            foreach ($machineJobs as $job) {
                $remainingHours = $this->getEffectiveRemainingHours($job);
                $segments = $this->computeJobSegments($machineModel['days'], $remainingHours, $daysLimit);

                $machineModel['jobs'][] = array(
                    'id' => (int) $job['rowid'],
                    'fg' => $job['fg'],
                    'works_order_no' => $job['works_order_no'],
                    'estimated_hours' => (float) $job['estimated_hours'],
                    'remaining_hours_override' => $job['remaining_hours_override'],
                    'remaining_hours_used' => $remainingHours,
                    'sort_order' => (int) $job['sort_order'],
                    'segments' => $segments,
                );
            }

            $result['machines'][] = $machineModel;
        }

        return $result;
    }

    /**
     * Compute planning day boundaries.
     * Each planning day runs from 06:30 to 06:30 next calendar day.
     *
     * @param string $startDay   Start date YYYY-MM-DD
     * @param int    $daysLimit  Number of days
     * @return array             Array of planning day info
     */
    private function computePlanningDays($startDay, $daysLimit)
    {
        $days = array();
        $baseTs = strtotime($startDay);

        for ($i = 0; $i < $daysLimit; $i++) {
            $dayTs = $baseTs + ($i * 86400);
            $date = date('Y-m-d', $dayTs);

            // Planning day starts at 06:30 on $date
            $startTs = strtotime($date . ' 06:30:00');
            // Ends at 06:30 on next calendar day
            $endTs = $startTs + 86400;

            $days[] = array(
                'index' => $i,
                'date' => $date,
                'start_ts' => $startTs,
                'end_ts' => $endTs,
            );
        }

        return $days;
    }

    /**
     * Load machines (workstations) for a group.
     * Uses Dolibarr workstation table, filters by label containing group keyword.
     *
     * @param string $groupCode Group code (forming/trimming)
     * @return array            Machines indexed by ID or error
     */
    private function loadMachines($groupCode)
    {
        // Detect workstation table (Dolibarr uses different names in versions)
        $wsTables = array(
            MAIN_DB_PREFIX . 'workstation_workstation',
            MAIN_DB_PREFIX . 'workstation',
            MAIN_DB_PREFIX . 'mrp_workstation',
        );

        $wsTable = null;
        foreach ($wsTables as $tbl) {
            $sqlchk = "SELECT 1 as ok FROM information_schema.TABLES
                       WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '" . $this->db->escape($tbl) . "'";
            $reschk = $this->db->query($sqlchk);
            if ($reschk) {
                $o = $this->db->fetch_object($reschk);
                $this->db->free($reschk);
                if ($o && !empty($o->ok)) {
                    $wsTable = $tbl;
                    break;
                }
            }
        }

        if (!$wsTable) {
            return array('error' => 'Workstation table not found.');
        }

        // Filter keyword based on group
        $keyword = ($groupCode === 'trimming') ? 'trim' : 'form';

        $sql = "SELECT rowid, ref, label
                FROM " . $wsTable . "
                WHERE (LOWER(label) LIKE '%" . $this->db->escape($keyword) . "%'
                   OR LOWER(ref) LIKE '%" . $this->db->escape($keyword) . "%')
                ORDER BY ref ASC";

        $resql = $this->db->query($sql);
        if (!$resql) {
            return array('error' => 'Database error loading workstations: ' . $this->db->lasterror());
        }

        $machines = array();
        while ($obj = $this->db->fetch_object($resql)) {
            $wsId = (int) $obj->rowid;
            $machines[$wsId] = array(
                'id' => $wsId,
                'ref' => $obj->ref,
                'label' => $obj->label,
                'group_code' => $groupCode,
            );
        }
        $this->db->free($resql);

        return $machines;
    }

    /**
     * Load base capacity for machines.
     *
     * @param array  $machineIds Machine IDs
     * @param string $startDay   Not used for base capacity
     * @param int    $daysLimit  Not used for base capacity
     * @return array             Capacity by machine ID
     */
    private function loadCapacities($machineIds, $startDay, $daysLimit)
    {
        if (empty($machineIds)) {
            return array();
        }

        $sql = "SELECT fk_machine, hours_per_day
                FROM " . MAIN_DB_PREFIX . "planning_capacity
                WHERE fk_machine IN (" . implode(',', array_map('intval', $machineIds)) . ")";

        $resql = $this->db->query($sql);
        if (!$resql) {
            return array();
        }

        $capacities = array();
        while ($obj = $this->db->fetch_object($resql)) {
            $capacities[(int) $obj->fk_machine] = (float) $obj->hours_per_day;
        }
        $this->db->free($resql);

        return $capacities;
    }

    /**
     * Load capacity overrides for machines and date range.
     *
     * @param array  $machineIds Machine IDs
     * @param string $startDay   Start date
     * @param int    $daysLimit  Number of days
     * @return array             Overrides by machine ID and date
     */
    private function loadCapacityOverrides($machineIds, $startDay, $daysLimit)
    {
        if (empty($machineIds)) {
            return array();
        }

        $endDay = date('Y-m-d', strtotime($startDay) + ($daysLimit * 86400));

        $sql = "SELECT fk_machine, override_date, hours_override
                FROM " . MAIN_DB_PREFIX . "planning_capacity_override
                WHERE fk_machine IN (" . implode(',', array_map('intval', $machineIds)) . ")
                  AND override_date >= '" . $this->db->escape($startDay) . "'
                  AND override_date < '" . $this->db->escape($endDay) . "'";

        $resql = $this->db->query($sql);
        if (!$resql) {
            return array();
        }

        $overrides = array();
        while ($obj = $this->db->fetch_object($resql)) {
            $mid = (int) $obj->fk_machine;
            $date = $obj->override_date;
            if (!isset($overrides[$mid])) {
                $overrides[$mid] = array();
            }
            $overrides[$mid][$date] = (float) $obj->hours_override;
        }
        $this->db->free($resql);

        return $overrides;
    }

    /**
     * Load downtimes for machines within planning days.
     *
     * @param array $machineIds   Machine IDs
     * @param array $planningDays Planning day boundaries
     * @return array              Downtime hours by machine and date
     */
    private function loadDowntimes($machineIds, $planningDays)
    {
        if (empty($machineIds) || empty($planningDays)) {
            return array();
        }

        $firstStart = $planningDays[0]['start_ts'];
        $lastEnd = $planningDays[count($planningDays) - 1]['end_ts'];

        $sql = "SELECT fk_machine, start_ts, end_ts
                FROM " . MAIN_DB_PREFIX . "planning_downtime
                WHERE fk_machine IN (" . implode(',', array_map('intval', $machineIds)) . ")
                  AND start_ts < " . (int) $lastEnd . "
                  AND end_ts > " . (int) $firstStart;

        $resql = $this->db->query($sql);
        if (!$resql) {
            return array();
        }

        $rawDowntimes = array();
        while ($obj = $this->db->fetch_object($resql)) {
            $rawDowntimes[] = array(
                'fk_machine' => (int) $obj->fk_machine,
                'start_ts' => (int) $obj->start_ts,
                'end_ts' => (int) $obj->end_ts,
            );
        }
        $this->db->free($resql);

        // Distribute downtime hours to planning days
        $downtimes = array();
        foreach ($rawDowntimes as $dt) {
            $mid = $dt['fk_machine'];
            if (!isset($downtimes[$mid])) {
                $downtimes[$mid] = array();
            }

            foreach ($planningDays as $day) {
                // Calculate overlap between downtime and planning day
                $overlapStart = max($dt['start_ts'], $day['start_ts']);
                $overlapEnd = min($dt['end_ts'], $day['end_ts']);
                if ($overlapEnd > $overlapStart) {
                    $hours = ($overlapEnd - $overlapStart) / 3600;
                    $date = $day['date'];
                    if (!isset($downtimes[$mid][$date])) {
                        $downtimes[$mid][$date] = 0;
                    }
                    $downtimes[$mid][$date] += $hours;
                }
            }
        }

        return $downtimes;
    }

    /** @var array Debug info from last loadJobs call */
    public $debugJobsInfo = array();

    /**
     * Load planning jobs assigned to machines.
     * Uses llx_planning_job table (future work, not MOs).
     *
     * @param array  $machineIds Machine IDs (workstation rowids)
     * @param string $groupCode  Group code (forming/trimming)
     * @return array             Jobs grouped by machine ID
     */
    private function loadJobs($machineIds, $groupCode)
    {
        global $conf;

        if (empty($machineIds)) {
            $this->debugJobsInfo = array('sql' => 'empty_machineIds', 'count' => 0, 'entity' => $conf->entity);
            return array();
        }

        $sql = "SELECT rowid, fk_workstation, fk_mo,
                       estimated_hours, remaining_hours_override, sort_order,
                       fg, works_order_no
                FROM " . MAIN_DB_PREFIX . "planning_job
                WHERE fk_workstation IN (" . implode(',', array_map('intval', $machineIds)) . ")
                  AND entity = " . (int) $conf->entity . "
                  AND group_code = '" . $this->db->escape($groupCode) . "'
                  AND status IN ('planned', 'needs_ruleset', 'incomplete_data')
                ORDER BY fk_workstation ASC, sort_order ASC, rowid ASC";

        $this->debugJobsInfo = array(
            'sql' => $sql,
            'entity' => $conf->entity,
            'machineIds' => $machineIds,
        );

        $resql = $this->db->query($sql);
        if (!$resql) {
            $this->debugJobsInfo['count'] = 'query_failed';
            $this->debugJobsInfo['error'] = $this->db->lasterror();
            return array();
        }

        $this->debugJobsInfo['count'] = $this->db->num_rows($resql);

        $jobs = array();
        while ($obj = $this->db->fetch_object($resql)) {
            $mid = (int) $obj->fk_workstation;
            if (!isset($jobs[$mid])) {
                $jobs[$mid] = array();
            }
            $jobs[$mid][] = array(
                'rowid' => $obj->rowid,
                'fk_mo' => $obj->fk_mo,
                'estimated_hours' => $obj->estimated_hours,
                'remaining_hours_override' => $obj->remaining_hours_override,
                'sort_order' => $obj->sort_order,
                'fg' => $obj->fg,
                'works_order_no' => $obj->works_order_no,
            );
        }
        $this->db->free($resql);

        return $jobs;
    }

    /**
     * Get effective remaining hours for a job.
     * Uses remaining_hours_override if set, otherwise estimated_hours.
     *
     * @param array $job Job data
     * @return float     Remaining hours
     */
    private function getEffectiveRemainingHours($job)
    {
        if ($job['remaining_hours_override'] !== null && $job['remaining_hours_override'] !== '') {
            return (float) $job['remaining_hours_override'];
        }
        return (float) $job['estimated_hours'];
    }

    /**
     * Compute segments for a job based on available capacity.
     * Fills available hours day by day until job is complete.
     *
     * @param array $days           Machine days with available_hours
     * @param float $remainingHours Total hours needed
     * @param int   $daysLimit      Max days
     * @return array                Segments with day_index and hours
     */
    private function computeJobSegments($days, $remainingHours, $daysLimit)
    {
        $segments = array();
        $hoursLeft = $remainingHours;

        for ($dayIdx = 0; $dayIdx < $daysLimit && $hoursLeft > 0; $dayIdx++) {
            if (!isset($days[$dayIdx])) {
                continue;
            }

            $available = $days[$dayIdx]['available_hours'];
            if ($available <= 0) {
                continue;
            }

            $used = min($available, $hoursLeft);
            $segments[] = array(
                'day_index' => $dayIdx,
                'date' => $days[$dayIdx]['date'],
                'hours' => round($used, 2),
            );

            $hoursLeft -= $used;
            // Reduce available for next job (tracked externally if needed)
            $days[$dayIdx]['available_hours'] -= $used;
        }

        // Mark if job extends beyond timeline
        if ($hoursLeft > 0) {
            $segments[] = array(
                'day_index' => -1,
                'date' => null,
                'hours' => round($hoursLeft, 2),
                'overflow' => true,
            );
        }

        return $segments;
    }
}
