# CLAUDE.md

Guidance for **Claude Code (claude.ai/code)** when working in this repository.

---

## 1) Project Overview

This repository contains a **custom Dolibarr ERP module** named **Planning** (`/custom/planning/`).  
It provides manufacturing planning views for **MRP Manufacturing Orders (MO)** grouped by **Working Station**, including a **Planning Board** with auto-refresh.

### Key Pages
- `planning.php` — table view grouped by working station (live MOs)
- `board.php` — **Planning Board** (primary interactive view)
- `completed.php` — completed/finished view
- `board_data.php` — AJAX endpoint used by `board.php` auto-refresh

### Current Implementation Style (IMPORTANT)
The current board implementation is **server-rendered HTML**:
- `board_data.php` returns JSON: `{ "clock": "...", "html": "<div>...</div>" }`
- JavaScript replaces the board DOM with the returned `html` and re-initializes tooltips.

**Do not refactor to a full JSON/card renderer unless explicitly requested.**  
Enhancements (like *late alert icon*) should be added in the current architecture first.

---

## 2) Technology & Constraints

- PHP 7.3+ (Dolibarr conventions)
- Vanilla JavaScript (ES5; keep compatibility; avoid ES6+)
- CSS3
- MySQL/MariaDB through Dolibarr `$db` abstraction

No npm, no Composer, no bundlers.

---

## 3) Repository Structure (actual)

```
planning/
  board.php
  board_data.php
  completed.php
  planning.php
  core/modules/modPlanning.class.php
  css/
    board.css
    planning.css
  js/
    board.js
```

---

## 4) Permissions / Access Control

This module uses `rights_class = 'planning'` and requires:

- `$user->rights->planning->read`

All pages and endpoints must enforce access checks:
- `accessforbidden()` in pages
- `403` JSON error in AJAX endpoint

---

## 5) Data Model Used by This Module

### Primary Dolibarr tables
- `llx_mrp_mo` (Manufacturing Orders)
- `llx_mrp_mo_extrafields` (custom extrafields used by this module)
  - `job_number`
  - `working_station`
- `llx_product` (product ref/label for MO product)
- `llx_stock_mouvement` (used to compute produced quantity)
- `llx_bom_bom` (used for BOM note tooltip)

### Working Station source (IMPORTANT)
**Working station is stored on MO extrafields**:
- `llx_mrp_mo_extrafields.working_station`

It may contain:
- a numeric ID (rowid) pointing to workstation tables, OR
- a free-text label/code

Workstation lookup (label/ref) currently comes from:
- `llx_workstation_workstation` (preferred)
Other files also probe fallback tables:
- `llx_workstation`
- `llx_mrp_workstation`

**Do NOT assume `workstation_id` / `workstation_label` are part of the MO card.**
They are a *lane/column context* resolved from extrafields + lookup table.

---

## 6) Board Rendering & Auto-Refresh

### `board.php`
- Renders the initial board HTML server-side.
- Includes `css/board.css` and `js/board.js`.
- Provides controls (station filter, show draft, refresh interval).
- Uses tooltip helpers to avoid broken tooltips after DOM replacement.

### `board_data.php`
- Returns JSON with `clock` and `html`.
- Builds board HTML grid (`.pl-board-grid`) with columns (`.pl-col`) and cards (`.pl-card`).

### `js/board.js`
- Handles fullscreen toggling and board theme switching:
  - `pl-board-dark` when fullscreen
  - `pl-board-light` otherwise
- Handles auto-refresh timer and AJAX fetch of `board_data.php`
- Re-initializes tooltips after DOM replacement

**When modifying the board UI, you must ensure the auto-refresh still works.**

---

## 7) Tooltip System (do not break)

This module intentionally avoids relying only on Dolibarr AJAX tooltips after refresh.

Current approach:
- Prefer TipTip (`.classfortooltip`, `.pl-job-tooltip`) where available
- Re-init Dolibarr tooltip JS after refresh as a fallback
- Extra CSRF-token injection helpers exist to keep `/core/ajax/ajaxtooltip.php` working after refresh

Rules:
- Do not remove tooltip re-init logic in `board.js`
- If adding new icons/links, ensure they have correct tooltip binding (TipTip or Dolibarr)

---

## 8) Theme Rules (Light vs Fullscreen Dark)

- Normal (non-fullscreen): must respect Dolibarr theme colors (light)
- Fullscreen: intentionally uses a **dark** board theme

Implementation:
- JS toggles `.pl-board-dark` / `.pl-board-light`
- CSS must be written so fullscreen does not leak into normal Dolibarr UI

Avoid:
- hard global overrides on `body` that affect other Dolibarr pages
- excessive `!important`

---

## 9) CSS Rules (STRICT)

- Avoid `!important` unless absolutely necessary
- Keep selectors scoped to the module:
  - prefer `.planning-page`, `.pl-board`, `.pl-card`, `.pl-col`, etc.
- Do not change Dolibarr global typography/colors

---

## 10) Late Alert Icon (Planned Feature)

Goal: show a **small late/overdue icon** on MO cards, similar to Dolibarr MO UI.

### Where to implement (current architecture)
Since cards are server-rendered:
- compute `is_late` in PHP while building each card in `board.php` and `board_data.php`
- render a small icon inside `.pl-card-top` (or near MO ref) when late

### Default late definition (match Dolibarr intent)
A card is late when:
- MO is not finished/cancelled (status not in produced/cancelled)
- `date_end_planned` exists
- `date_end_planned` < current server time

### UI requirements
- Icon must be subtle (small)
- Icon must have a tooltip showing planned end date and “Late”
- Reuse Dolibarr pictos where possible (do not add new icon libraries)

### Stability requirement
Late icon MUST survive:
- auto-refresh DOM replacement
- fullscreen mode
- tooltips after refresh

---

## 11) Performance Rules

- Do not introduce N+1 queries per card
- Any extra data needed for late icon must come from already available fields (`date_end_planned`, `status`)
- Keep board_data response lightweight; avoid expensive per-card object fetch if not required

---

## 12) Development Workflow

- Deploy to: `dolibarr/htdocs/custom/planning/`
- Enable module in Dolibarr admin
- Test in a dev instance (not production)

No build steps.

---

## 13) Language Note

You MAY write instructions and comments in **Polish or English**.

Recommended:
- High-level notes / TODO: English OK
- Code (variables, functions): English
- Keep consistency within a file

---

## 14) Safe Change Policy

When implementing changes:
- Prefer **small, reversible commits**
- Avoid refactors that touch all pages at once
- Keep `board.php` and `board_data.php` output consistent (same card structure)
