# 🔍 Code Locations - Quick Reference

## Modified Files

### 1. [ajax/add_job.php](ajax/add_job.php)

#### Exception Handler
**Lines: 57-97**
- Original: Generic "Server error processing request" message
- Updated: Returns actual exception message + debug fields (if POST + debug enabled)
- Key check: `if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode())`

#### Shutdown Handler  
**Lines: 99-135**
- Original: Generic error message
- Updated: Returns actual error message + debug fields (if POST + debug enabled)
- Key check: `if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode())`

#### Main Try-Catch Block
**Lines: 381-407**
- Original: `outputJson(array('error' => $e->getMessage()), 500);`
- Updated: Full error logging + conditional debug fields
- Changes:
  - Lines 383-388: Extract error details (message, file, line)
  - Lines 390-394: Log to dol_syslog with full context
  - Lines 398-406: Build response with conditional debug fields

#### Debug Fields Added (all 3 locations)
```php
'debug_message' => $errorMsg;
'debug_file' => $errorFile;
'debug_line' => $errorLine;
'db_error' => $dbErr;  // (if any)
```

#### Key Security Check
```php
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isDebugMode())
```
- Ensures debug only for POST (not GET)
- Ensures debug only when explicitly enabled

---

### 2. [js/timeline.js](js/timeline.js)

#### Form Submit Error Handler
**Lines: 1150-1169**
- Original: Simple error display
- Updated: Enhanced error display with debug info if present

**Changes:**
```javascript
// Before (lines 1151-1152):
if (data.error) {
    if (errorDiv) errorDiv.textContent = data.error;

// After (lines 1151-1161):
if (data.error) {
    var errorMsg = data.error;
    // Append debug info if available
    if (data.debug_message || data.debug_file) {
        errorMsg += '\n\nDebug Info:\n';
        if (data.debug_message) errorMsg += '  Message: ' + data.debug_message + '\n';
        if (data.debug_file) errorMsg += '  File: ' + data.debug_file + '\n';
        if (data.debug_line) errorMsg += '  Line: ' + data.debug_line + '\n';
        if (data.db_error) errorMsg += '  DB Error: ' + data.db_error + '\n';
    }
    if (errorDiv) errorDiv.textContent = errorMsg;
```

#### Key Features
- Gracefully handles missing debug fields (checks `if` before using)
- Appends debug info only if present
- No breaking changes (works with or without debug fields)

---

## Related Files (No Changes)

### [timeline.php](timeline.php)
**Line: ~100** - Where to add `?debug=1` for testing
```php
// TO ENABLE DEBUG, CHANGE:
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php';

// TO:
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php?debug=1';
```

---

## Function Reference

### isDebugMode() Function
**Location:** [ajax/add_job.php](ajax/add_job.php) lines 30-42
- Checks `GETPOST('debug', 'int') == 1`
- Checks `$conf->global->PLANNING_DEBUG`
- Checks user is admin
- Returns boolean

### outputJson() Function  
**Location:** [ajax/add_job.php](ajax/add_job.php) lines 145-153
- Sets HTTP response code
- Outputs JSON with `JSON_UNESCAPED_UNICODE`
- Exits script
- Used for all responses (success and error)

### dol_syslog() Function
**Dolibarr Core Function**
- Parameters: `dol_syslog(message, level)`
- Levels: `LOG_DEBUG`, `LOG_INFO`, `LOG_WARNING`, `LOG_ERR`
- Used at lines: 62, 68, 72, 106, 109, 389-394
- Logs to: `/documents/dolibarr.log`

---

## HTTP Status Codes

### 403 - Forbidden
- **Trigger:** Invalid/missing CSRF token
- **Location:** Line 227 in [ajax/add_job.php](ajax/add_job.php)
- **Response:** `{'error': 'Invalid or missing security token.'}`
- **Debug fields:** None (not POST validation error)

### 405 - Method Not Allowed
- **Trigger:** Non-POST request
- **Location:** Line 234 in [ajax/add_job.php](ajax/add_job.php)
- **Response:** `{'error': 'Invalid request method.'}`
- **Debug fields:** None

### 400 - Bad Request
- **Trigger:** Validation errors (invalid group code, missing workstation, etc.)
- **Locations:** Lines 242-298
- **Response:** `{'error': 'descriptive message'}`
- **Debug fields:** None (validation errors, not runtime errors)

### 500 - Internal Server Error
- **Trigger:** Runtime exception
- **Locations:** Exception handler (line 97), Shutdown handler (line 135), Main catch (line 407)
- **Response:** 
  - Without debug: `{'error': 'message'}`
  - With debug: `{'error': 'message', 'debug_message': '...', 'debug_file': '...', 'debug_line': 123, 'db_error': '...'}`
- **Debug fields:** Conditional (POST + isDebugMode() only)

### 200 - OK
- **Trigger:** Job created successfully
- **Location:** Line 373 in [ajax/add_job.php](ajax/add_job.php)
- **Response:** `{'success': true, 'message': 'Job created successfully.', 'job_id': 123}`
- **Debug fields:** None

---

## Error Paths

### Path 1: Exception During Processing
```
Line 389-394: Extract error details
Line 390-394: Log to syslog
Line 398-406: Build response with conditional debug
Line 407: outputJson($response, 500)
```

### Path 2: Exception During File Include
```
Line 57-97: Global exception handler
Line 62: Log to syslog
Line 82: Check POST + debug
Line 85-92: Add debug fields if enabled
Line 97: Return JSON
```

### Path 3: Fatal PHP Error
```
Line 99-135: Shutdown handler
Line 104: Log to syslog
Line 122: Check POST + debug
Line 125-132: Add debug fields if enabled
Line 135: Return JSON
```

---

## Testing Points

### Test 1: Verify Exception Handler Works
- File: [ajax/add_job.php](ajax/add_job.php)
- Lines: 57-97
- Trigger: Throw exception during rate lookup
- Expected: Return JSON with debug fields

### Test 2: Verify Main Catch Works
- File: [ajax/add_job.php](ajax/add_job.php)
- Lines: 381-407
- Trigger: Missing product rate during calculation
- Expected: Return JSON with debug fields

### Test 3: Verify JavaScript Display Works
- File: [js/timeline.js](js/timeline.js)
- Lines: 1150-1169
- Trigger: POST response with debug_message field
- Expected: Modal shows error + debug info

### Test 4: Verify GET Never Shows Debug
- File: [ajax/add_job.php](ajax/add_job.php)
- Lines: 82, 122
- Trigger: Direct GET to add_job.php?debug=1
- Expected: Returns 403 (no debug fields)

---

## Configuration Locations

### Enable Debug
**Option 1:** [timeline.php](timeline.php) line ~100
```php
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php?debug=1';
```

**Option 2:** Dolibarr config
```php
$conf->global->PLANNING_DEBUG = 1;
```

**Option 3:** Automatic for admins
- Checked in isDebugMode() at [ajax/add_job.php](ajax/add_job.php) line 36

### Disable Debug
**Option 1:** Remove from [timeline.php](timeline.php)
```php
$addJobEndpoint = DOL_URL_ROOT . '/custom/planning/ajax/add_job.php';
```

**Option 2:** Dolibarr config
```php
$conf->global->PLANNING_DEBUG = 0;
```

---

## Logging Locations

### Server Logs
**File:** `/documents/dolibarr.log`
- All errors logged here (always)
- Format: `Planning Add Job Error: ... in /path/file.php:123`
- Level: `LOG_ERR`

### dol_syslog Calls
- Line 62: Exception handler main error
- Line 68: Exception handler stack trace
- Line 72: Exception handler DB error
- Line 106: Shutdown handler fatal error
- Line 109: Shutdown handler DB error
- Line 390-391: Main catch block error + file:line
- Line 393-394: Main catch block DB error

---

## Security Checkpoints

1. **Line 224-226:** CSRF token validation (unchanged)
2. **Line 82, 122:** POST method check (debug fields only for POST)
3. **Line 82, 122, 400:** isDebugMode() check (opt-in)
4. **Line 145-153:** JSON encoding (consistent escaping)

---

## Quick Navigation

| What | File | Line(s) |
|------|------|---------|
| Exception handler | [ajax/add_job.php](ajax/add_job.php) | 57-97 |
| Shutdown handler | [ajax/add_job.php](ajax/add_job.php) | 99-135 |
| Main catch block | [ajax/add_job.php](ajax/add_job.php) | 381-407 |
| JS error display | [js/timeline.js](js/timeline.js) | 1150-1169 |
| Enable debug (temp) | [timeline.php](timeline.php) | ~100 |
| isDebugMode() | [ajax/add_job.php](ajax/add_job.php) | 30-42 |
| outputJson() | [ajax/add_job.php](ajax/add_job.php) | 145-153 |

---

## Summary

- **2 files modified**
- **4 locations with changes** (3 in PHP, 1 in JS)
- **~40 lines of code**
- **All changes are minimal and focused**
- **Easy to find, review, and maintain**

**Start debugging:** Go to [timeline.php](timeline.php) line ~100 and add `?debug=1`
