<?php
require '../../main.inc.php';

require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT.'/mrp/class/mo.class.php';
require_once __DIR__.'/lib/planning_board_render.lib.php';

if (empty($user->rights->planning) || empty($user->rights->planning->read)) accessforbidden();

// Ensure translations used by getNomUrl() (MO/Product) are available also during AJAX refresh.
$langs->loadLangs(array("main", "mrp", "products"));

// Add CSRF token into Dolibarr ajax tooltip payload (data-params) so
// /core/ajax/ajaxtooltip.php works after AJAX auto-refresh.
function pl_get_csrf_token()
{
    // Do NOT call newToken() during AJAX refresh, it may rotate tokens.
    if (!empty($_SESSION['newtoken'])) return $_SESSION['newtoken'];
    if (function_exists('newToken')) {
        $t = newToken();
        if (!empty($t)) return $t;
    }
    return '';
}


function pl_add_token_to_ajaxtooltip_params($html)
{
    $token = pl_get_csrf_token();
    if (empty($token) || empty($html)) return $html;

    // Support both single and double quoted data-params
    $pattern = '/\sdata-params=("[^"]*"|\'[^\']*\')/i';
    if (!preg_match($pattern, $html, $m)) return $html;

    $raw = $m[1];
    $quote = $raw[0];
    $json = substr($raw, 1, -1);
    // getNomUrl() often HTML-escapes quotes inside data-params (e.g. &quot;).
    // Decode entities before json_decode, then re-escape on output.
    $jsonDecoded = html_entity_decode($json, ENT_QUOTES);
    $params = json_decode($jsonDecoded, true);
    if (!is_array($params)) return $html;
    if (empty($params['action'])) $params['action'] = 'fetch';
    $params['token'] = $token;
    $newjson = json_encode($params);
    if ($newjson === false) return $html;

    // Preserve original escaping level (if any)
    $newjsonEsc = htmlspecialchars($newjson, ENT_QUOTES);
    $replacement = ' data-params='.$quote.$newjsonEsc.$quote;
    return preg_replace($pattern, $replacement, $html, 1);
}

// -----------------------------------------------------------------------------
// HTML tooltip helpers (TipTip via classfortooltip)
// We avoid Dolibarr AJAX tooltips (classforajaxtooltip) because they break after
// DOM replacement during auto-refresh and show title="tocomplete" placeholders.
// -----------------------------------------------------------------------------

function pl_tt_attr($html)
{
    // For safe usage inside title="..." attribute.
    $html = str_replace("\r", '', (string) $html);
    $html = str_replace("\n", '', (string) $html);
    return str_replace('"', '&quot;', $html);
}

function pl_make_tiptip_link($href, $innerHtml, $tooltipHtml, $extraClass = '')
{
    $cls = trim('classfortooltip '.(string)$extraClass);
    return '<a href="'.dol_escape_htmltag($href).'" class="'.$cls.'" data-html="true" title="'.pl_tt_attr($tooltipHtml).'">'.$innerHtml.'</a>';
}

function pl_render_mo_tooltip_html($moObj, $j)
{
    $refVal = '';
    if (is_object($moObj) && !empty($moObj->ref)) $refVal = $moObj->ref;
    elseif (!empty($j['ref'])) $refVal = $j['ref'];
    $ref = dol_escape_htmltag($refVal);
    $qty = isset($j['qty']) ? (float)$j['qty'] : 0;
    $done = isset($j['qty_done']) ? (float)$j['qty_done'] : 0;
    $left = max(0, $qty - $done);
    $ws = !empty($j['ws_ref']) ? dol_escape_htmltag($j['ws_ref']) : '';
    $prod = !empty($j['prod_ref']) ? dol_escape_htmltag($j['prod_ref']) : '';
    $prodLabel = !empty($j['prod_label']) ? dol_escape_htmltag($j['prod_label']) : '';
    $job = !empty($j['job_number']) ? dol_escape_htmltag($j['job_number']) : '';
    $startTxt = !empty($j['start']) ? dol_print_date($j['start'], 'dayhour') : '';
    $endTxt   = !empty($j['end']) ? dol_print_date($j['end'], 'dayhour') : '';

    $html = '<b><span class="fas fa-cubes paddingright" style="color:#a69944"></span>'.$ref.'</b>';
    if ($prod) $html .= '<br><span class="fas fa-cube paddingright" style="color:#a69944"></span>'.$prod.($prodLabel ? ' - '.$prodLabel : '');
    if ($job)  $html .= '<br><b>Job:</b> '.$job;
    if ($ws)   $html .= '<br><b>Station:</b> '.$ws;
    if ($qty || $done || $left) $html .= '<br><b>Qty:</b> '.$qty.' &nbsp; <b>Done:</b> '.$done.' &nbsp; <b>Left:</b> '.$left;
    if ($startTxt) $html .= '<br><b>Start:</b> '.$startTxt;
    if ($endTxt)   $html .= '<br><b>End:</b> '.$endTxt;
    return $html;
}

function pl_render_product_tooltip_html($pobj, $j)
{
    $refVal = '';
    if (is_object($pobj) && !empty($pobj->ref)) $refVal = $pobj->ref;
    elseif (!empty($j['prod_ref'])) $refVal = $j['prod_ref'];
    $ref = dol_escape_htmltag($refVal);

    $label = '';
    if (is_object($pobj) && !empty($pobj->label)) $label = dol_escape_htmltag($pobj->label);
    elseif (!empty($j['prod_label'])) $label = dol_escape_htmltag($j['prod_label']);
    $moRef = !empty($j['ref']) ? dol_escape_htmltag($j['ref']) : '';
    $job = !empty($j['job_number']) ? dol_escape_htmltag($j['job_number']) : '';
    $ws = !empty($j['ws_ref']) ? dol_escape_htmltag($j['ws_ref']) : '';
    $qty = isset($j['qty']) ? (float)$j['qty'] : 0;

    $html = '<b><span class="fas fa-cube paddingright" style="color:#a69944"></span>'.$ref.'</b>';
    if ($label) $html .= '<br>'.$label;
    if ($moRef) $html .= '<br><b>MO:</b> '.$moRef;
    if ($job)   $html .= '<br><b>Job:</b> '.$job;
    if ($ws)    $html .= '<br><b>Station:</b> '.$ws;
    if ($qty)   $html .= '<br><b>Qty:</b> '.$qty;
    return $html;
}

$pagetitle = "Planning Board";
$stationFilter = trim(GETPOST('station', 'alphanohtml'));
$showDraft = (int) GETPOST('showdraft', 'int');
$ajax = (int) GETPOST('ajax', 'int');
// Auto-refresh is handled in JS. We keep a server-side default only for initial UI state.
// 0 = Off, otherwise seconds.
$autorefresh_raw = GETPOST('refresh', 'alpha');
$autorefresh = ($autorefresh_raw === '' ? 60 : (int) $autorefresh_raw);
if ($autorefresh < 0) $autorefresh = 60;

$arrayofcss = array();
$morecss = array('/custom/planning/css/board.css');
$morejs  = array('/custom/planning/js/board.js');
if (!$ajax) llxHeader('', 'Planning Board', '', '', 0, 0, $morejs, $morecss);
// --- Workstation map (rowid => ref/label)
$workstationMap = array();       // id => "Label"
$workstationRefMap = array();    // id => "REF"
$wsTables = array(MAIN_DB_PREFIX.'workstation_workstation', MAIN_DB_PREFIX.'workstation', MAIN_DB_PREFIX.'mrp_workstation');
foreach ($wsTables as $wst) {
    // Detect table existence safely
    $sqlchk = "SELECT 1 as ok FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '".$db->escape($wst)."'";
    $reschk = $db->query($sqlchk);
    $ok = ($reschk && ($o = $db->fetch_object($reschk)) && !empty($o->ok));
    if (!$ok) continue;

    $sqlws = "SELECT rowid, ref, label FROM ".$wst;
    $resws = $db->query($sqlws);
    if ($resws) {
        while ($ws = $db->fetch_object($resws)) {
            $id = (int) $ws->rowid;
            $ref = trim((string) $ws->ref);
            $label = trim((string) $ws->label);
            $workstationRefMap[$id] = $ref;
//            $display = ($label !== '' ? $label : $ref);
            $display = '';
            if ($label !== '' && $ref !== '' && $label !== $ref) $display = $label.' - '.$ref;
            elseif ($label !== '') $display = $label;
            else $display = $ref;
            $workstationMap[$id] = $display;
        }
        break;
    }
}

// --- Build list of available stations (only live: Validated+In progress (+Draft if toggle))
$availableStations = array(); // raw => label
$sqlst = "SELECT DISTINCT ef.working_station as ws_raw
          FROM ".MAIN_DB_PREFIX."mrp_mo mo
          LEFT JOIN ".MAIN_DB_PREFIX."mrp_mo_extrafields ef ON ef.fk_object = mo.rowid
          WHERE mo.entity IN (".getEntity('mrp_mo').")
            AND ef.working_station IS NOT NULL
            AND ef.working_station <> ''
            AND mo.status IN (1,2".($showDraft ? ",0" : "").")
          ORDER BY ef.working_station ASC";
$resst = $db->query($sqlst);
if ($resst) {
    while ($s = $db->fetch_object($resst)) {
        $raw = (string) $s->ws_raw;
        $lbl = $raw;
        if (ctype_digit($raw)) {
            $id = (int) $raw;
            if (isset($workstationMap[$id])) $lbl = $workstationMap[$id];
        }
        $availableStations[$raw] = $lbl;
    }
    asort($availableStations, SORT_NATURAL | SORT_FLAG_CASE);
}

// --- Main SQL (live view only): statuses Validated + In progress (+ Draft via toggle)
$sql = "SELECT
            mo.rowid,
            mo.ref,
            mo.status,
            mo.qty,
            mo.fk_product,
            mo.date_start_planned,
            mo.date_end_planned,
            (SELECT COALESCE(SUM(CASE WHEN sm.value > 0 THEN sm.value ELSE 0 END),0)
             FROM ".MAIN_DB_PREFIX."stock_mouvement sm
             WHERE sm.origintype IN ('mo','mrp_mo')
               AND sm.fk_origin = mo.rowid
               AND sm.type_mouvement IN (2,3)
            ) as qty_done,
            p.ref as prod_ref,
            p.label as prod_label,
            ef.job_number,
            ef.working_station,
            mo.fk_bom,
            b.note_public as bom_note_public
        FROM ".MAIN_DB_PREFIX."mrp_mo as mo
        LEFT JOIN ".MAIN_DB_PREFIX."mrp_mo_extrafields as ef ON ef.fk_object = mo.rowid
        LEFT JOIN ".MAIN_DB_PREFIX."product as p ON p.rowid = mo.fk_product
        LEFT JOIN ".MAIN_DB_PREFIX."bom_bom as b ON b.rowid = mo.fk_bom
        WHERE mo.entity IN (".getEntity('mrp_mo').")
          AND mo.status IN (1,2".($showDraft ? ",0" : "").")";

if ($stationFilter !== '') {
    $sql .= " AND ef.working_station = '".$db->escape($stationFilter)."' ";
}

$sql .= " ORDER BY ef.working_station ASC, mo.date_start_planned ASC, mo.ref ASC";

$res = $db->query($sql);
if (!$res) {
    print '<div class="error">'.$db->lasterror().'</div>';
    llxFooter();
    exit;
}

// Group by station
$stations = array();
while ($obj = $db->fetch_object($res)) {
    $wsKey = trim((string) $obj->working_station);
    if ($wsKey === '') $wsKey = '(none)';
    if (!isset($stations[$wsKey])) $stations[$wsKey] = array();

    $start = $db->jdate($obj->date_start_planned);
    $end   = $db->jdate($obj->date_end_planned);

    $stations[$wsKey][] = array(
        'rowid' => (int) $obj->rowid,
        'ref' => $obj->ref,
        'status' => (int) $obj->status,
        'qty' => (float) $obj->qty,
        'qty_done' => (float) $obj->qty_done,
        'fk_product' => (int) $obj->fk_product,
        'prod_ref' => $obj->prod_ref,
        'prod_label' => $obj->prod_label,
        'start' => $start,
        'end' => $end,
        'job_number' => $obj->job_number,
        'fk_bom' => (int) $obj->fk_bom,
        'bom_note_public' => $obj->bom_note_public,
    );
}

function ws_ref_label($wsKey, $workstationRefMap, $workstationMap)
{
    $wsKey = trim((string) $wsKey);
    if ($wsKey === '' || $wsKey === '(none)') return $wsKey === '' ? '(none)' : $wsKey;

    // Stored as workstation rowid
    if (ctype_digit($wsKey)) {
        $id = (int) $wsKey;
        if (!empty($workstationMap[$id])) return (string) $workstationMap[$id];
        if (!empty($workstationRefMap[$id])) return (string) $workstationRefMap[$id];
        return $wsKey;
    }

    // Stored as workstation ref (e.g. FM7)
    foreach ($workstationRefMap as $id => $ref) {
        if (strcasecmp((string) $ref, $wsKey) === 0) {
            if (!empty($workstationMap[(int)$id])) return (string) $workstationMap[(int)$id];
            return (string) $ref;
        }
    }

    return $wsKey;
}
function status_badge($status) {
    // Dolibarr-ish labels
    $map = array(
        0 => array('Draft','pl-status-draft'),
        1 => array('Validated','pl-status-validated'),
        2 => array('In progress','pl-status-inprogress'),
        3 => array('Produced','pl-status-produced'),
        9 => array('Cancelled','pl-status-cancelled'),
    );
    if (!isset($map[$status])) return '<span class="pl-status pl-status-unknown">?</span>';
    return '<span class="pl-status '.$map[$status][1].'">'.$map[$status][0].'</span>';
}

// --- AJAX refresh endpoint (returns JSON with refreshed board grid HTML) ---
if ($ajax) {
    ob_start();
    print '<div class="pl-board-grid" id="plBoardGrid">';
    foreach ($stations as $wsKey => $jobs) {
        $wsRef = ws_ref_label($wsKey, $workstationRefMap, $workstationMap);
        $count = count($jobs);

        print '<div class="pl-col">';
        print '<div class="pl-col-head">'.dol_escape_htmltag($wsRef).'<span class="pl-col-count">Jobs ('.$count.')</span></div>';

        foreach ($jobs as $j) {
            $qty = (float)$j['qty'];
            $done = (float)$j['qty_done'];
            $left = max(0, $qty - $done);
            $pct = ($qty > 0) ? min(100, max(0, ($done / $qty) * 100)) : 0;

            $startTxt = $j['start'] ? dol_print_date($j['start'], 'dayhour') : '';
            $endTxt   = $j['end'] ? dol_print_date($j['end'], 'dayhour') : '';
            $duration = '';
            if (!empty($j['start']) && !empty($j['end'])) {
                $seconds = ((int)$j['end']) - ((int)$j['start']);
                if ($seconds > 0) $duration = convertSecondToTime($seconds, 'allhourmin');
            } elseif (!empty($j['start']) && (int)$j['status'] === 2) {
                $seconds = time() - ((int)$j['start']);
                if ($seconds > 0) $duration = convertSecondToTime($seconds, 'allhourmin');
            }

            // MO link (TipTip HTML tooltip)
            $moUrl = DOL_URL_ROOT.'/mrp/mo_card.php?id='.$j['rowid'];
            $moInner = '<span class="fas fa-cubes paddingright" style=" color: #a69944;"></span>'.dol_escape_htmltag($j['ref']);
            $moLink = pl_make_tiptip_link($moUrl, $moInner, pl_render_mo_tooltip_html(null, $j), 'nowraponall');

            // Product link (TipTip HTML tooltip)
            $prodRefTxt = dol_escape_htmltag($j['prod_ref']);
            $prodLabelTxt = '';
            if (!empty($j['prod_label'])) $prodLabelTxt = dol_escape_htmltag($j['prod_label']);

            $prodUrl = (!empty($j['fk_product']) ? (DOL_URL_ROOT.'/product/card.php?id='.(int)$j['fk_product']) : '#');
            $prodInner = '<span class="fas fa-cube paddingright" style=" color: #a69944;"></span><span class="aaa">'.$prodRefTxt.'</span>';
            $prodLink = pl_make_tiptip_link($prodUrl, $prodInner, pl_render_product_tooltip_html(null, $j), 'nowraponall');
            if ($prodLabelTxt) $prodLink .= ' - '.$prodLabelTxt;

            // Job link tooltip from BOM note (formatted)
            $jobTxt = (string)$j['job_number'];
            $jobLink = dol_escape_htmltag($jobTxt);
            $bomId = (int)$j['fk_bom'];
            if ($bomId > 0) {
                $noteRaw = (string)$j['bom_note_public'];
                $noteRaw = stripcslashes($noteRaw);
                $noteRaw = str_replace(array('\\r\\n','\\n','\\r'), "\n", $noteRaw);
                $noteRaw = str_replace(array("\r\n","\r"), "\n", $noteRaw);
                $noteRaw = trim($noteRaw);
                $titleHtml = dol_escape_htmltag($noteRaw);
                $titleHtml = str_replace(array('\\r\\n','\\n','\\r'), "<br>", $titleHtml);
                $titleHtml = str_replace(array("\r\n","\n","\r"), "<br>", $titleHtml);
                $urlNote = DOL_URL_ROOT.'/bom/bom_note.php?id='.$bomId;
                $jobLink = '<a href="'.dol_escape_htmltag($urlNote).'" class="classfortooltip pl-job-tooltip" data-html="true" title="'.$titleHtml.'">'.dol_escape_htmltag($jobTxt).'</a>';
            }

            print pl_render_mo_card_html($j, $moLink, $prodLink, $jobLink);
        }

        print '</div>'; // col
    }
    print '</div>'; // grid
    $gridHtml = ob_get_clean();

    $payload = array(
        'html' => $gridHtml,
        'clock' => dol_print_date(dol_now(), 'dayhoursec'),
    );
    header('Content-Type: application/json; charset=utf-8');
    print json_encode($payload);
    exit;
}

print '<div class="pl-board-page" id="plBoardPage">';

// Top bar
print '<div class="pl-board-top">';
print '<div class="pl-board-title">'.$pagetitle.'</div>';

print '<form class="pl-board-filters" method="GET" action="'.dol_escape_htmltag($_SERVER['PHP_SELF']).'">';
print '<select name="station" class="minwidth200">';
print '<option value="">All stations</option>';
foreach ($availableStations as $raw => $lbl) {
    $sel = ($stationFilter !== '' && (string)$stationFilter === (string)$raw) ? ' selected' : '';
    print '<option value="'.dol_escape_htmltag($raw).'"'.$sel.'>'.dol_escape_htmltag($lbl).'</option>';
}
print '</select>';

$chk = $showDraft ? ' checked' : '';
print '<label class="pl-board-check"><input type="checkbox" name="showdraft" value="1"'.$chk.'> Show Draft</label>';
print '<label class="pl-board-check"><input type="checkbox" id="plLateOnly" value="1"> Late only</label>';
print '<label class="pl-board-check"><input type="checkbox" id="plHideEmpty" value="1"> Hide empty</label>';

print '<select name="refresh" class="minwidth100">';
foreach (array(0=>'Off',30=>'30s',60=>'60s',120=>'2m',300=>'5m') as $sec=>$txt) {
    $sel = ((int)$autorefresh === (int)$sec) ? ' selected' : '';
    print '<option value="'.((int)$sec).'"'.$sel.'>'.$txt.'</option>';
}
print '</select>';

print '<button class="butAction" type="submit">Apply</button>';
print '<button class="butAction" type="button" id="plFullscreen">Fullscreen</button>';
print '</form>';

print '<div class="pl-board-meta">';
print '<span id="plClock"></span>';
print '<span id="plLastUpdate" style="margin-left:10px;">Last update: -</span>';
// Auto-refresh status is handled purely in JS to keep fullscreen stable and avoid duplicate labels.
print '</div>';
print '</div>';

// Two-section layout (Forming left, Trimming right)
print '<div class="pl-board-layout" id="plBoardLayout">';
print '<div class="pl-board-section" id="plSectionForming"><div class="pl-section-title">Forming</div><div class="pl-board-grid"></div></div>';
print '<div class="pl-board-section" id="plSectionTrimming"><div class="pl-section-title">Trimming</div><div class="pl-board-grid"></div></div>';
print '</div>';

// Hidden source grid for columns (JS moves them to sections)
print '<div class="pl-board-grid" id="plBoardGrid" style="display:none">';
foreach ($stations as $wsKey => $jobs) {
    $wsRef = ws_ref_label($wsKey, $workstationRefMap, $workstationMap);
    $count = count($jobs);

    print '<div class="pl-col">';
    print '<div class="pl-col-head">'.dol_escape_htmltag($wsRef).'<span class="pl-col-count">Jobs ('.$count.')</span></div>';

    foreach ($jobs as $j) {
        $qty = (float)$j['qty'];
        $done = (float)$j['qty_done'];
        $left = max(0, $qty - $done);
        $pct = ($qty > 0) ? min(100, max(0, ($done / $qty) * 100)) : 0;

        $startTxt = $j['start'] ? dol_print_date($j['start'], 'dayhour') : '';
        $endTxt   = $j['end'] ? dol_print_date($j['end'], 'dayhour') : '';
        $duration = '';
        if (!empty($j['start']) && !empty($j['end'])) {
            $sec = ((int)$j['end']) - ((int)$j['start']);
            if ($sec > 0) $duration = convertSecondToTime($sec, 'allhourmin');
        } elseif (!empty($j['start']) && (int)$j['status'] === 2) {
            $sec = time() - ((int)$j['start']);
            if ($sec > 0) $duration = convertSecondToTime($sec, 'allhourmin');
        }

        // MO link (TipTip HTML tooltip)
        $moUrl = DOL_URL_ROOT.'/mrp/mo_card.php?id='.$j['rowid'];
        $moInner = '<span class="fas fa-cubes paddingright" style=" color: #a69944;"></span>'.dol_escape_htmltag($j['ref']);
        $moLink = pl_make_tiptip_link($moUrl, $moInner, pl_render_mo_tooltip_html(null, $j), 'nowraponall');

        // Product link (TipTip HTML tooltip)
        $prodRefTxt = dol_escape_htmltag($j['prod_ref']);
        $prodLabelTxt = '';
        if (!empty($j['prod_label'])) $prodLabelTxt = dol_escape_htmltag($j['prod_label']);

        $prodUrl = (!empty($j['fk_product']) ? (DOL_URL_ROOT.'/product/card.php?id='.(int)$j['fk_product']) : '#');
        $prodInner = '<span class="fas fa-cube paddingright" style=" color: #a69944;"></span><span class="aaa">'.$prodRefTxt.'</span>';
        $prodLink = pl_make_tiptip_link($prodUrl, $prodInner, pl_render_product_tooltip_html(null, $j), 'nowraponall');
        if ($prodLabelTxt) $prodLink .= ' - '.$prodLabelTxt;

        $jobTxt = (string)$j['job_number'];
        $jobLink = dol_escape_htmltag($jobTxt);
        $bomId = (int)$j['fk_bom'];
        if ($bomId > 0) {
            $noteRaw = (string)$j['bom_note_public'];
            $noteRaw = stripcslashes($noteRaw);
            $noteRaw = str_replace(array('\\r\\n','\\n','\\r'), "\n", $noteRaw);
            $noteRaw = str_replace(array("\r\n","\r"), "\n", $noteRaw);
            $noteRaw = trim($noteRaw);
            $titleHtml = dol_escape_htmltag($noteRaw);
            $titleHtml = str_replace(array('\\r\\n','\\n','\\r'), "<br>", $titleHtml);
            $titleHtml = str_replace(array("\r\n","\n","\r"), "<br>", $titleHtml);
            $urlNote = DOL_URL_ROOT.'/bom/bom_note.php?id='.$bomId;
            $jobLink = '<a href="'.dol_escape_htmltag($urlNote).'" class="classfortooltip pl-job-tooltip" data-html="true" title="'.$titleHtml.'">'.dol_escape_htmltag($jobTxt).'</a>';
        }

        print pl_render_mo_card_html($j, $moLink, $prodLink, $jobLink);
    }

    print '</div>'; // col
}
print '</div>'; // grid

print '</div>'; // page

print '';

print '';

llxFooter();
