<?php
/**
 * Planning - Update Job AJAX endpoint
 * Updates an existing planning job.
 *
 * Required POST parameters:
 *   token          - Dolibarr CSRF token
 *   job_id         - ID of the job to update
 *
 * Optional POST parameters:
 *   fk_workstation - workstation ID
 *   group_code     - 'forming' or 'trimming'
 *   estimated_hours - estimated hours
 *   date_start      - start datetime (YYYY-MM-DDTHH:MM or YYYY-MM-DD HH:MM:SS)
 *   date_end        - end datetime (YYYY-MM-DDTHH:MM or YYYY-MM-DD HH:MM:SS)
 *   status          - job status
 *   notes           - notes text
 */

// ===== GLOBAL ERROR HANDLERS - Must be at top =====
// Prevent direct output before headers
ob_start();

/**
 * Check if debug mode should be enabled
 * Debug mode is ON if: ?debug=1, PLANNING_DEBUG global, or user is admin
 */
function isDebugMode() {
    global $user;
    if (GETPOST('debug', 'int') == 1) {
        return true;
    }
    if (!empty($GLOBALS['conf']->global->PLANNING_DEBUG)) {
        return true;
    }
    if (!empty($user) && !empty($user->admin)) {
        return true;
    }
    return false;
}

// Global error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    global $db;
    $msg = "PHP Error [$errno] in $errfile:$errline - $errstr";
    dol_syslog('Planning Update Job: ' . $msg, LOG_ERR);
    return false;
});

// Global exception handler
set_exception_handler(function($ex) {
    global $db, $user;
    $msg = "Exception: " . $ex->getMessage() . " in " . $ex->getFile() . ":" . $ex->getLine();
    dol_syslog('Planning Update Job: ' . $msg, LOG_ERR);
    
    // Log stack trace
    if (!empty($ex->getTraceAsString())) {
        dol_syslog('Planning Update Job: Stack trace: ' . $ex->getTraceAsString(), LOG_ERR);
    }
    
    // Log DB error if applicable
    if (!empty($db)) {
        $dbErr = $db->lasterror();
        if (!empty($dbErr)) {
            dol_syslog('Planning Update Job: DB error: ' . $dbErr, LOG_ERR);
        }
    }
    
    ob_clean();
    http_response_code(500);
    
    $response = [
        'success' => false,
        'error' => 'Server error processing request'
    ];
    
    // Include debug info if enabled
    if (isDebugMode()) {
        $response['debug'] = $ex->getMessage() . ' @ ' . $ex->getFile() . ':' . $ex->getLine();
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                $response['db_error'] = $dbErr;
            }
        }
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
});

// Shutdown handler - catch any fatal errors
register_shutdown_function(function() {
    global $db, $user;
    $error = error_get_last();
    if ($error !== null && ($error['type'] & (E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR))) {
        $msg = "Fatal Error [" . $error['type'] . "] in " . $error['file'] . ":" . $error['line'] . " - " . $error['message'];
        dol_syslog('Planning Update Job: ' . $msg, LOG_ERR);
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                dol_syslog('Planning Update Job: DB error: ' . $dbErr, LOG_ERR);
            }
        }
        ob_clean();
        http_response_code(500);
        
        $response = [
            'success' => false,
            'error' => 'Server error processing request'
        ];
        
        // Include debug info if enabled
        if (isDebugMode()) {
            $response['debug'] = $error['message'] . ' @ ' . $error['file'] . ':' . $error['line'];
            if (!empty($db)) {
                $dbErr = $db->lasterror();
                if (!empty($dbErr)) {
                    $response['db_error'] = $dbErr;
                }
            }
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    }
});

require '../../../main.inc.php';

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// Clear any buffered output
ob_end_clean();

/**
 * Output JSON response and exit.
 *
 * @param array $data Response data
 * @param int   $code HTTP status code
 */
function outputJson($data, $code = 200)
{
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Load production rate from product extrafields
 * Uses Dolibarr Product class with array_options (extrafields with options_ prefix)
 *
 * @param DoliDB $db Database handler
 * @param int    $fk_product Product ID
 * @param string $groupCode 'forming' or 'trimming'
 * @return float|null Rate in pieces/hour, or null if not found/invalid
 */
function loadProductRate($db, $fk_product, $groupCode)
{
    if (!$fk_product) {
        return null;
    }

    try {
        require_once(DOL_DOCUMENT_ROOT . '/product/class/product.class.php');
        require_once(DOL_DOCUMENT_ROOT . '/core/class/extrafields.class.php');

        $product = new Product($db);
        $res = $product->fetch((int)$fk_product);
        if ($res <= 0) {
            dol_syslog('Planning: Product fetch failed for ID ' . $fk_product, LOG_WARNING);
            return null;
        }

        // Load extrafields into array_options
        $extrafields = new ExtraFields($db);
        $extralabels = $extrafields->fetch_name_optionals_label($product->table_element, true);
        if (count($extralabels) > 0) {
            $product->fetch_optionals($product->id, $extralabels);
        }

        // Get rate from array_options with options_ prefix
        $fieldname = ($groupCode === 'forming') ? 'options_qty_per_hour_form' : 'options_qty_per_hour_trim';

        $arrayOpts = $product->array_options ?? array();
        if (isset($arrayOpts[$fieldname]) && !empty($arrayOpts[$fieldname])) {
            $rate = (float)$arrayOpts[$fieldname];
            dol_syslog('Planning: Product #' . $fk_product . ' rate (' . $fieldname . ') = ' . $rate, LOG_DEBUG);
            return $rate > 0 ? $rate : null;
        }

        dol_syslog('Planning: Product #' . $fk_product . ' missing rate field ' . $fieldname, LOG_DEBUG);
        return null;
    } catch (Exception $e) {
        dol_syslog('Planning: Error loading product rate: ' . $e->getMessage(), LOG_ERR);
        return null;
    }
}

/**
 * Get buffer minutes based on group code
 *
 * @param string $groupCode 'forming' or 'trimming'
 * @return int Buffer in minutes
 */
function getBufferForGroupCode($groupCode)
{
    if ($groupCode === 'forming') {
        return 60; // 1 hour for forming
    } elseif ($groupCode === 'trimming') {
        return 30; // 30 minutes for trimming
    }
    return 60; // Default fallback
}

// Check CSRF token
$token = GETPOST('token', 'aZ09');
$sessionToken = isset($_SESSION['newtoken']) ? $_SESSION['newtoken'] : (isset($_SESSION['token']) ? $_SESSION['token'] : '');
if (empty($token) || $token !== $sessionToken) {
    outputJson(array('error' => 'Invalid or missing security token.'), 403);
}

// Check permissions - require editjob right
if (empty($user->rights->planning) || empty($user->rights->planning->editjob)) {
    outputJson(array('error' => 'Access denied. Missing planning editjob permission.'), 403);
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    outputJson(array('error' => 'Invalid request method.'), 405);
}

// Get job_id (required)
$job_id = GETPOST('job_id', 'int');
if (empty($job_id) || $job_id <= 0) {
    outputJson(array('error' => 'Missing or invalid job_id.'), 400);
}

// Verify job exists and get current data
$sqlCheck = "SELECT rowid, fk_workstation, group_code, status FROM " . MAIN_DB_PREFIX . "planning_job WHERE rowid = " . (int) $job_id;
$resCheck = $db->query($sqlCheck);
if (!$resCheck || !($currentJob = $db->fetch_object($resCheck))) {
    outputJson(array('error' => 'Job not found.'), 404);
}

// Get optional parameters
$fk_workstation = GETPOST('fk_workstation', 'int');
$fk_product = GETPOST('fk_product', 'int');
$group_code = GETPOST('group_code', 'aZ09');
$qty = GETPOST('qty', 'alpha');
$date_start = GETPOST('date_start', 'alphanohtml');
$status = GETPOST('status', 'aZ09');
$notes = GETPOST('notes', 'restricthtml');

// Build update fields array
$updateFields = array();

// Workstation
if (!empty($fk_workstation) && $fk_workstation > 0) {
    // Validate workstation exists
    $sqlWsCheck = "SELECT rowid FROM " . MAIN_DB_PREFIX . "workstation_workstation WHERE rowid = " . (int) $fk_workstation;
    $resWsCheck = $db->query($sqlWsCheck);
    if (!$resWsCheck || !$db->fetch_object($resWsCheck)) {
        outputJson(array('error' => 'Selected workstation does not exist.'), 400);
    }
    $updateFields[] = "fk_workstation = " . (int) $fk_workstation;
}

// Product (safe - handles pre and post-migration states)
if (GETPOSTISSET('fk_product')) {
    // Try to update fk_product if provided
    // Will fail gracefully if column doesn't exist yet (before migration)
    if (!empty($fk_product) && $fk_product > 0) {
        $updateFields[] = "fk_product = " . (int) $fk_product;
    } else {
        $updateFields[] = "fk_product = NULL";
    }
}

// Group code and Workstation are read-only in edit mode - IGNORE if provided in POST
// These cannot be changed once job is created
if (!empty($group_code) || !empty($fk_workstation)) {
    dol_syslog('Planning: Ignoring POST group_code/fk_workstation (read-only fields)', LOG_DEBUG);
    // Do not add to updateFields
}

// Status
if (!empty($status)) {
    $validStatuses = array('planned', 'in_progress', 'paused', 'completed', 'cancelled');
    if (!in_array($status, $validStatuses)) {
        outputJson(array('error' => 'Invalid status value.'), 400);
    }
    $updateFields[] = "status = '" . $db->escape($status) . "'";
}

// Quantity
if ($qty !== '' && $qty !== null) {
    $qty = str_replace(',', '.', $qty);
    $qty = (float) $qty;
    if ($qty > 0) {
        $updateFields[] = "qty = " . (float) $qty;
    }
}

// Date start
if ($date_start !== '' && $date_start !== null) {
    $date_start = str_replace('T', ' ', $date_start);
    if (strlen($date_start) === 16) {
        $date_start .= ':00';
    }
    $ts = strtotime($date_start);
    if ($ts !== false) {
        $updateFields[] = "date_start = '" . $db->escape(date('Y-m-d H:i:s', $ts)) . "'";
    }
} elseif (GETPOSTISSET('date_start')) {
    // Explicitly clearing the field
    $updateFields[] = "date_start = NULL";
}

// Notes (allow empty to clear)
if (GETPOSTISSET('notes')) {
    $updateFields[] = "notes = '" . $db->escape($notes) . "'";
}

// Check if there's anything to update
if (empty($updateFields)) {
    outputJson(array('error' => 'No fields to update.'), 400);
}

// Always update modifier info
$updateFields[] = "fk_user_modif = " . (int) $user->id;
$updateFields[] = "tms = NOW()";

// TG2.1: Calculate date_end if we have the necessary data (date_start, qty, fk_product, group_code)
// Fetch current job data to merge with updates
$sqlFetch = "SELECT fk_product, qty, group_code, date_start FROM " . MAIN_DB_PREFIX . "planning_job WHERE rowid = " . (int) $job_id;
$resFetch = $db->query($sqlFetch);
$jobData = $db->fetch_object($resFetch);

// Determine final values for date_end calculation
$finalProduct = !empty($fk_product) && $fk_product > 0 ? $fk_product : (isset($jobData->fk_product) ? $jobData->fk_product : null);
$finalQty = !empty($qty) && (float) $qty > 0 ? (float) $qty : (isset($jobData->qty) ? (float) $jobData->qty : null);
$finalGroupCode = !empty($group_code) ? $group_code : (isset($jobData->group_code) ? $jobData->group_code : null);

// VALIDATION: fk_product is required (must have valid product after merge)
if (!$finalProduct || (int) $finalProduct <= 0) {
    outputJson(array('error' => 'Product (FG) is required.'), 400);
}

// Get start date: from update or current
$finalDateStart = null;
if (!empty($date_start)) {
    $date_start = str_replace('T', ' ', $date_start);
    if (strlen($date_start) === 16) {
        $date_start .= ':00';
    }
    $ts = strtotime($date_start);
    if ($ts !== false) {
        $finalDateStart = date('Y-m-d H:i:s', $ts);
    }
} elseif (isset($jobData->date_start) && !empty($jobData->date_start)) {
    $finalDateStart = $jobData->date_start;
}

// Calculate date_end if all required data is available
if ($finalDateStart && $finalQty && $finalQty > 0 && $finalProduct && $finalProduct > 0 && $finalGroupCode && in_array($finalGroupCode, array('forming', 'trimming'))) {
    try {
        $rate = loadProductRate($db, $finalProduct, $finalGroupCode);
        if ($rate && $rate > 0) {
            $durationHours = $finalQty / $rate;
            $bufferMinutes = getBufferForGroupCode($finalGroupCode);
            $durationSeconds = ($durationHours * 3600) + ($bufferMinutes * 60);

            $startTs = strtotime($finalDateStart);
            $endTs = $startTs + $durationSeconds;
            $calculatedDateEnd = date('Y-m-d H:i:s', $endTs);

            // Only set date_end if it's a new calculation (not a manual override)
            $updateFields[] = "date_end = '" . $db->escape($calculatedDateEnd) . "'";
        }
    } catch (Exception $e) {
        // If rate calculation fails, just skip date_end update (don't fail the whole update)
        dol_syslog('Planning Update Job: Warning - could not calculate date_end: ' . $e->getMessage(), LOG_WARNING);
    }
}

// Begin transaction
$db->begin();

try {
    $sql = "UPDATE " . MAIN_DB_PREFIX . "planning_job SET " . implode(', ', $updateFields) . " WHERE rowid = " . (int) $job_id;

    $result = $db->query($sql);
    if (!$result) {
        throw new Exception('Database error: ' . $db->lasterror());
    }

    $db->commit();

    outputJson(array(
        'success' => true,
        'message' => 'Job updated successfully.',
        'job_id' => (int) $job_id
    ), 200);

} catch (Exception $e) {
    $db->rollback();
    outputJson(array('error' => $e->getMessage()), 500);
}