<?php
/**
 * Planning - Delete Job AJAX endpoint
 * Deletes a planning job (hard delete).
 *
 * Required POST parameters:
 *   token          - Dolibarr CSRF token
 *   job_id         - ID of the job to delete
 */

// ===== GLOBAL ERROR HANDLERS - Must be at top =====
// Prevent direct output before headers
ob_start();

/**
 * Check if debug mode should be enabled
 * Debug mode is ON if: ?debug=1, PLANNING_DEBUG global, or user is admin
 */
function isDebugMode() {
    global $user;
    if (GETPOST('debug', 'int') == 1) {
        return true;
    }
    if (!empty($GLOBALS['conf']->global->PLANNING_DEBUG)) {
        return true;
    }
    if (!empty($user) && !empty($user->admin)) {
        return true;
    }
    return false;
}

// Global error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    global $db;
    $msg = "PHP Error [$errno] in $errfile:$errline - $errstr";
    dol_syslog('Planning Delete Job: ' . $msg, LOG_ERR);
    return false;
});

// Global exception handler
set_exception_handler(function($ex) {
    global $db, $user;
    $msg = "Exception: " . $ex->getMessage() . " in " . $ex->getFile() . ":" . $ex->getLine();
    dol_syslog('Planning Delete Job: ' . $msg, LOG_ERR);
    
    // Log stack trace
    if (!empty($ex->getTraceAsString())) {
        dol_syslog('Planning Delete Job: Stack trace: ' . $ex->getTraceAsString(), LOG_ERR);
    }
    
    // Log DB error if applicable
    if (!empty($db)) {
        $dbErr = $db->lasterror();
        if (!empty($dbErr)) {
            dol_syslog('Planning Delete Job: DB error: ' . $dbErr, LOG_ERR);
        }
    }
    
    ob_clean();
    http_response_code(500);
    
    $response = [
        'success' => false,
        'error' => 'Server error processing request'
    ];
    
    // Include debug info if enabled
    if (isDebugMode()) {
        $response['debug'] = $ex->getMessage() . ' @ ' . $ex->getFile() . ':' . $ex->getLine();
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                $response['db_error'] = $dbErr;
            }
        }
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
});

// Shutdown handler - catch any fatal errors
register_shutdown_function(function() {
    global $db, $user;
    $error = error_get_last();
    if ($error !== null && ($error['type'] & (E_ERROR | E_PARSE | E_CORE_ERROR | E_COMPILE_ERROR))) {
        $msg = "Fatal Error [" . $error['type'] . "] in " . $error['file'] . ":" . $error['line'] . " - " . $error['message'];
        dol_syslog('Planning Delete Job: ' . $msg, LOG_ERR);
        if (!empty($db)) {
            $dbErr = $db->lasterror();
            if (!empty($dbErr)) {
                dol_syslog('Planning Delete Job: DB error: ' . $dbErr, LOG_ERR);
            }
        }
        ob_clean();
        http_response_code(500);
        
        $response = [
            'success' => false,
            'error' => 'Server error processing request'
        ];
        
        // Include debug info if enabled
        if (isDebugMode()) {
            $response['debug'] = $error['message'] . ' @ ' . $error['file'] . ':' . $error['line'];
            if (!empty($db)) {
                $dbErr = $db->lasterror();
                if (!empty($dbErr)) {
                    $response['db_error'] = $dbErr;
                }
            }
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    }
});

require '../../../main.inc.php';

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// Clear any buffered output
ob_end_clean();

/**
 * Output JSON response and exit.
 *
 * @param array $data Response data
 * @param int   $code HTTP status code
 */
function outputJson($data, $code = 200)
{
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

// Check CSRF token
$token = GETPOST('token', 'aZ09');
$sessionToken = isset($_SESSION['newtoken']) ? $_SESSION['newtoken'] : (isset($_SESSION['token']) ? $_SESSION['token'] : '');
if (empty($token) || $token !== $sessionToken) {
    outputJson(array('error' => 'Invalid or missing security token.'), 403);
}

// Check permissions - require deletejob right
if (empty($user->rights->planning) || empty($user->rights->planning->deletejob)) {
    outputJson(array('error' => 'Access denied. Missing planning deletejob permission.'), 403);
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    outputJson(array('error' => 'Invalid request method.'), 405);
}

// Get job_id (required)
$job_id = GETPOST('job_id', 'int');
if (empty($job_id) || $job_id <= 0) {
    outputJson(array('error' => 'Missing or invalid job_id.'), 400);
}

// Verify job exists
$sqlCheck = "SELECT rowid FROM " . MAIN_DB_PREFIX . "planning_job WHERE rowid = " . (int) $job_id;
$resCheck = $db->query($sqlCheck);
if (!$resCheck || !$db->fetch_object($resCheck)) {
    outputJson(array('error' => 'Job not found.'), 404);
}

// Begin transaction
$db->begin();

try {
    // Hard delete the job
    $sql = "DELETE FROM " . MAIN_DB_PREFIX . "planning_job WHERE rowid = " . (int) $job_id;

    $result = $db->query($sql);
    if (!$result) {
        throw new Exception('Database error: ' . $db->lasterror());
    }

    $db->commit();

    outputJson(array(
        'success' => true,
        'message' => 'Job deleted successfully.',
        'job_id' => (int) $job_id
    ), 200);

} catch (Exception $e) {
    $db->rollback();
    outputJson(array('error' => $e->getMessage()), 500);
}