/**
 * Planning Timeline - Read-only visualization
 */

(function() {
  'use strict';

  var config = {};
  var timelineData = null;
  var selectedJob = null;

  // DOM helpers
  function $(id) { return document.getElementById(id); }

  /**
   * Compute day column width based on container and day count.
   * Returns clamped value: min 55px, max 110px.
   * @param {number} dayCount Number of days
   * @param {number} containerWidth Container width in pixels (optional)
   * @returns {number} Day width in pixels
   */
  function getDayWidth(dayCount, containerWidth) {
    var machineColWidth = 140;
    var defaultWidth = 80;
    if (!containerWidth || containerWidth <= 0) {
      containerWidth = 800; // fallback
    }
    if (!dayCount || dayCount <= 0) {
      return defaultWidth;
    }
    var availableWidth = containerWidth - machineColWidth - 20; // scrollbar buffer
    var computedWidth = Math.floor(availableWidth / dayCount);
    return Math.max(55, Math.min(computedWidth, 110));
  }

  /**
   * Read config from data attributes on #plTimelinePage
   */
  function loadConfig() {
    var el = $('plTimelinePage');
    if (!el) return false;
    config.group = el.dataset.group || '';
    config.days = parseInt(el.dataset.days, 10) || 14;
    config.endpoint = el.dataset.endpoint || '';
    config.addjobEndpoint = el.dataset.addjobEndpoint || '';
    config.token = el.dataset.token || '';
    config.canAddjob = el.dataset.canAddjob === '1';
    return !!config.endpoint;
  }
  function esc(s) { return String(s).replace(/[&<>"']/g, function(c) { return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]; }); }

  /**
   * Fetch timeline data from endpoint
   */
  function fetchTimeline() {
    var url = config.endpoint + '?group=' + encodeURIComponent(config.group) +
              '&days=' + config.days + '&start=' + getTodayDate();

    var grid = $('plTimelineGrid');
    if (grid) grid.innerHTML = '<div class="pl-timeline-loading">Loading...</div>';

    fetch(url, { credentials: 'same-origin' })
      .then(function(r) { return r.json(); })
      .then(function(data) {
        if (data.error) {
          renderError(data.error);
          return;
        }
        timelineData = data;
        renderTimeline(data);
      })
      .catch(function(err) {
        renderError('Failed to load timeline: ' + err.message);
      });
  }

  /**
   * Get today's date as YYYY-MM-DD
   */
  function getTodayDate() {
    var d = new Date();
    return d.getFullYear() + '-' +
           String(d.getMonth() + 1).padStart(2, '0') + '-' +
           String(d.getDate()).padStart(2, '0');
  }

  /**
   * Format date for display
   */
  function formatDate(dateStr) {
    if (!dateStr) return '';
    var parts = dateStr.split('-');
    return parts[2] + '/' + parts[1];
  }

  /**
   * Render error message
   */
  function renderError(msg) {
    var grid = $('plTimelineGrid');
    if (grid) {
      grid.innerHTML = '<div class="pl-timeline-error">' + esc(msg) + '</div>';
    }
  }

  /**
   * Render the timeline grid
   */
  function renderTimeline(data) {
    var grid = $('plTimelineGrid');
    if (!grid) return;

    // Set CSS variables for consistent column sizing
    var dayCount = data.planning_days ? data.planning_days.length : data.days_limit;

    // Compute day width using helper function
    var panel = $('plTimelinePanel');
    var containerWidth = panel ? panel.clientWidth : 800;
    var dayWidth = getDayWidth(dayCount, containerWidth);

    grid.style.setProperty('--pl-day-count', dayCount);
    grid.style.setProperty('--pl-day-width', dayWidth + 'px');

    // Toggle compact mode for narrow columns (under 80px)
    grid.classList.toggle('pl-compact', dayWidth < 80);

    // Clear previous content completely
    grid.innerHTML = '';

    var html = '';

    // Day header row - use planning_days as single source of truth
    var planningDays = data.planning_days || [];
    html += '<div class="pl-tl-row pl-tl-header-row">';
    html += '<div class="pl-tl-machine-cell pl-tl-header-machine">Machine</div>';
    for (var i = 0; i < planningDays.length; i++) {
      var dayInfo = planningDays[i];
      var dayLabel = i === 0 ? 'Today' : 'D+' + i;
      var dateLabel = formatDate(dayInfo.date);
      html += '<div class="pl-tl-day-cell pl-tl-header-day" data-day="' + i + '">';
      html += '<div class="pl-tl-day-label">' + dayLabel + '</div>';
      html += '<div class="pl-tl-day-date">' + dateLabel + '</div>';
      html += '</div>';
    }
    html += '</div>';

    // Machine rows
    if (!data.machines || data.machines.length === 0) {
      html += '<div class="pl-tl-empty">No machines found for this group.</div>';
    } else {
      data.machines.forEach(function(machine) {
        html += renderMachineRow(machine, planningDays, dayWidth);
      });
    }

    grid.innerHTML = html;

    // Bind click events for job blocks
    grid.querySelectorAll('.pl-tl-job-block').forEach(function(block) {
      block.addEventListener('click', function(e) {
        e.stopPropagation(); // Prevent cell click
        selectJob(block.dataset.machineId, block.dataset.jobId);
      });
    });

    // Bind click events for day cells (add job on empty cell click)
    if (config.canAddjob) {
      grid.querySelectorAll('.pl-tl-machine-row .pl-tl-day-cell').forEach(function(cell) {
        cell.addEventListener('click', function() {
          var machineId = cell.dataset.machineId;
          var date = cell.dataset.date;
          if (machineId && date) {
            openAddJobModalWithPrefill(machineId, date);
          }
        });
      });
    }
  }

  /**
   * Render a single machine row
   * @param {Object} machine Machine data
   * @param {Array} planningDays Planning days array
   * @param {number} dayWidth Day column width in pixels
   */
  function renderMachineRow(machine, planningDays, dayWidth) {
    var html = '<div class="pl-tl-row pl-tl-machine-row" data-machine-id="' + machine.id + '">';

    // Machine label cell
    html += '<div class="pl-tl-machine-cell">';
    html += '<div class="pl-tl-machine-ref">' + esc(machine.ref) + '</div>';
    if (machine.label && machine.label !== machine.ref) {
      html += '<div class="pl-tl-machine-label">' + esc(machine.label) + '</div>';
    }
    html += '</div>';

    // Day cells with job segments - use planningDays.length as source of truth
    for (var dayIdx = 0; dayIdx < planningDays.length; dayIdx++) {
      var dayData = machine.days[dayIdx];
      var availableHours = dayData ? dayData.available_hours : 0;
      var maxHours = dayData ? dayData.effective_capacity : 24;
      var dayDate = planningDays[dayIdx] ? planningDays[dayIdx].date : '';

      html += '<div class="pl-tl-day-cell" data-day="' + dayIdx + '" data-date="' + dayDate + '" data-machine-id="' + machine.id + '">';

      // Find job segments for this day
      var segmentsForDay = [];
      machine.jobs.forEach(function(job) {
        job.segments.forEach(function(seg) {
          if (seg.day_index === dayIdx) {
            segmentsForDay.push({ job: job, segment: seg });
          }
        });
      });

      // Render job blocks
      segmentsForDay.forEach(function(item) {
        var job = item.job;
        var seg = item.segment;
        // Compute width in pixels: (hours / capacity) * dayWidth, clamped
        var ratio = maxHours > 0 ? (seg.hours / maxHours) : 0;
        var rawWidth = ratio * dayWidth;
        var blockWidth = Math.max(6, Math.min(rawWidth, dayWidth - 8));

        html += '<div class="pl-tl-job-block" ';
        html += 'data-machine-id="' + machine.id + '" ';
        html += 'data-job-id="' + job.id + '" ';
        html += 'style="width: ' + blockWidth.toFixed(0) + 'px;" ';
        var jobLabel = job.works_order_no || job.fg || ('Job#' + job.id);
        html += 'title="' + esc(jobLabel) + ' - ' + seg.hours.toFixed(1) + 'h / ' + maxHours.toFixed(1) + 'h capacity">';
        html += '<span class="pl-tl-job-ref">' + esc(jobLabel) + '</span>';
        html += '<span class="pl-tl-job-hours">' + seg.hours.toFixed(1) + 'h</span>';
        html += '</div>';
      });

      // Show available capacity indicator
      if (segmentsForDay.length === 0 && availableHours > 0) {
        html += '<div class="pl-tl-available" title="Available: ' + availableHours.toFixed(1) + 'h"></div>';
      }

      html += '</div>';
    }

    html += '</div>';
    return html;
  }

  /**
   * Select a job and show details in the card panel
   */
  function selectJob(machineId, jobId) {
    machineId = parseInt(machineId, 10);
    jobId = parseInt(jobId, 10);

    // Find machine and job
    var machine = null;
    var job = null;
    if (timelineData && timelineData.machines) {
      timelineData.machines.forEach(function(m) {
        if (m.id === machineId) {
          machine = m;
          m.jobs.forEach(function(j) {
            if (j.id === jobId) job = j;
          });
        }
      });
    }

    if (!job) return;

    selectedJob = { machine: machine, job: job };

    // Highlight selected block
    document.querySelectorAll('.pl-tl-job-block').forEach(function(b) {
      b.classList.remove('pl-tl-job-selected');
    });
    document.querySelectorAll('.pl-tl-job-block[data-job-id="' + jobId + '"]').forEach(function(b) {
      b.classList.add('pl-tl-job-selected');
    });

    // Render job card
    renderJobCard(machine, job);
  }

  /**
   * Render job details card
   */
  function renderJobCard(machine, job) {
    var card = $('plJobCard');
    if (!card) return;

    var jobTitle = job.works_order_no || job.fg || ('Job#' + job.id);
    var html = '<div class="pl-job-card-header">';
    html += '<div class="pl-job-card-title">' + esc(jobTitle) + '</div>';
    html += '<div class="pl-job-card-machine">' + esc(machine.ref) + '</div>';
    html += '</div>';

    html += '<div class="pl-job-card-body">';

    // Key info row - hours summary
    html += '<div class="pl-job-card-row">';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">' + (job.remaining_hours_used || 0).toFixed(1) + 'h</span>';
    html += '<span class="pl-job-card-stat-label">Remaining</span>';
    html += '</div>';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">' + (job.estimated_hours || 0).toFixed(1) + 'h</span>';
    html += '<span class="pl-job-card-stat-label">Estimated</span>';
    html += '</div>';
    html += '<div class="pl-job-card-stat">';
    html += '<span class="pl-job-card-stat-value">#' + (job.sort_order || 0) + '</span>';
    html += '<span class="pl-job-card-stat-label">Order</span>';
    html += '</div>';
    html += '</div>';

    // Override notice
    if (job.remaining_hours_override !== null && job.remaining_hours_override !== '') {
      html += '<div class="pl-job-card-notice">Hours override active</div>';
    }

    // Schedule segments
    html += '<div class="pl-job-card-section">';
    html += '<div class="pl-job-card-label">Schedule</div>';
    html += '<div class="pl-job-card-segments">';
    job.segments.forEach(function(seg) {
      if (seg.overflow) {
        html += '<div class="pl-job-card-seg pl-job-card-seg-overflow">Overflow: ' + seg.hours.toFixed(1) + 'h</div>';
      } else {
        var dayLabel = seg.day_index === 0 ? 'Today' : 'D+' + seg.day_index;
        html += '<div class="pl-job-card-seg">' + dayLabel + ': ' + seg.hours.toFixed(1) + 'h</div>';
      }
    });
    html += '</div>';
    html += '</div>';

    // Collapsible details
    html += '<details class="pl-job-card-details">';
    html += '<summary>More details</summary>';
    html += '<div class="pl-job-card-details-body">';
    if (job.fg) {
      html += '<div class="pl-job-card-detail"><span>FG:</span> ' + esc(job.fg) + '</div>';
    }
    if (job.works_order_no) {
      html += '<div class="pl-job-card-detail"><span>Works Order:</span> ' + esc(job.works_order_no) + '</div>';
    }
    html += '<div class="pl-job-card-detail"><span>Job ID:</span> ' + job.id + '</div>';
    html += '</div>';
    html += '</details>';

    html += '</div>'; // body

    card.innerHTML = html;
  }

  /**
   * Change days parameter
   */
  window.plTimelineChangeDays = function(days) {
    var url = new URL(window.location.href);
    url.searchParams.set('days', days);
    window.location.href = url.toString();
  };

  // ========== ADD JOB MODAL ==========

  /**
   * Format datetime for input[type=datetime-local]
   * @param {Date} date
   * @returns {string} YYYY-MM-DDTHH:MM
   */
  function formatDatetimeLocal(date) {
    var y = date.getFullYear();
    var m = String(date.getMonth() + 1).padStart(2, '0');
    var d = String(date.getDate()).padStart(2, '0');
    var hh = String(date.getHours()).padStart(2, '0');
    var mm = String(date.getMinutes()).padStart(2, '0');
    return y + '-' + m + '-' + d + 'T' + hh + ':' + mm;
  }

  /**
   * Update end time based on start time and estimated hours
   */
  function updateEndTime() {
    var startInput = $('plAddJobStart');
    var endInput = $('plAddJobEnd');
    var hoursInput = $('plAddJobHours');
    if (!startInput || !endInput || !hoursInput) return;
    if (!startInput.value) return;

    var startDate = new Date(startInput.value);
    var hours = parseFloat(hoursInput.value) || 1.0;
    var endDate = new Date(startDate.getTime() + hours * 60 * 60 * 1000);
    endInput.value = formatDatetimeLocal(endDate);
  }

  /**
   * Open the Add Job modal
   */
  function openAddJobModal() {
    var modal = $('plAddJobModal');
    if (!modal) return;
    modal.classList.add('pl-modal-open');
    // Reset form
    var form = $('plAddJobForm');
    if (form) form.reset();
    // Set group to current tab
    var groupSelect = $('plAddJobGroup');
    if (groupSelect) groupSelect.value = config.group;
    // Clear start/end
    var startInput = $('plAddJobStart');
    var endInput = $('plAddJobEnd');
    if (startInput) startInput.value = '';
    if (endInput) endInput.value = '';
    // Clear error
    var errorDiv = $('plAddJobError');
    if (errorDiv) errorDiv.textContent = '';
  }

  /**
   * Open the Add Job modal with prefilled values from cell click
   * @param {string} machineId Workstation ID
   * @param {string} date Date string YYYY-MM-DD
   */
  function openAddJobModalWithPrefill(machineId, date) {
    openAddJobModal();

    // Prefill workstation
    var wsSelect = $('plAddJobWorkstation');
    if (wsSelect) {
      wsSelect.value = machineId;
    }

    // Prefill start: date at 06:30 (planning day boundary)
    var startInput = $('plAddJobStart');
    if (startInput && date) {
      var startDate = new Date(date + 'T06:30:00');
      startInput.value = formatDatetimeLocal(startDate);
    }

    // Compute end based on estimated hours
    updateEndTime();
  }

  /**
   * Close the Add Job modal
   */
  function closeAddJobModal() {
    var modal = $('plAddJobModal');
    if (!modal) return;
    modal.classList.remove('pl-modal-open');
  }

  /**
   * Submit the Add Job form
   */
  function submitAddJobForm(e) {
    e.preventDefault();

    var form = $('plAddJobForm');
    var errorDiv = $('plAddJobError');
    var submitBtn = $('plModalSubmit');
    if (!form || !config.addjobEndpoint) return;

    // Clear error
    if (errorDiv) errorDiv.textContent = '';

    // Gather form data
    var formData = new FormData(form);
    formData.append('token', config.token);

    // Disable submit button
    if (submitBtn) {
      submitBtn.disabled = true;
      submitBtn.textContent = 'Adding...';
    }

    fetch(config.addjobEndpoint, {
      method: 'POST',
      credentials: 'same-origin',
      body: formData
    })
    .then(function(r) {
      return r.json().catch(function() {
        throw new Error('Invalid server response');
      });
    })
    .then(function(data) {
      if (data.error) {
        if (errorDiv) errorDiv.textContent = data.error;
        return;
      }
      if (!data.success) {
        if (errorDiv) errorDiv.textContent = 'Unknown error occurred';
        return;
      }
      // Success - close modal and refresh timeline
      closeAddJobModal();
      fetchTimeline();
    })
    .catch(function(err) {
      if (errorDiv) errorDiv.textContent = 'Request failed: ' + err.message;
    })
    .finally(function() {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = 'Add Job';
      }
    });
  }

  /**
   * Bind Add Job modal events
   */
  function bindAddJobEvents() {
    if (!config.canAddjob) return;

    var btnOpen = $('plBtnAddJob');
    var btnClose = $('plModalClose');
    var btnCancel = $('plModalCancel');
    var modal = $('plAddJobModal');
    var form = $('plAddJobForm');

    if (btnOpen) {
      btnOpen.addEventListener('click', openAddJobModal);
    }
    if (btnClose) {
      btnClose.addEventListener('click', closeAddJobModal);
    }
    if (btnCancel) {
      btnCancel.addEventListener('click', closeAddJobModal);
    }
    if (modal) {
      modal.addEventListener('click', function(e) {
        if (e.target === modal) closeAddJobModal();
      });
    }
    if (form) {
      form.addEventListener('submit', submitAddJobForm);
    }
    // ESC key to close
    document.addEventListener('keydown', function(e) {
      if (e.key === 'Escape' && modal && modal.classList.contains('pl-modal-open')) {
        closeAddJobModal();
      }
    });

    // Auto-update end time when start or hours change
    var startInput = $('plAddJobStart');
    var hoursInput = $('plAddJobHours');
    if (startInput) {
      startInput.addEventListener('change', updateEndTime);
    }
    if (hoursInput) {
      hoursInput.addEventListener('change', updateEndTime);
      hoursInput.addEventListener('input', updateEndTime);
    }
  }

  /**
   * Initialize on DOM ready
   */
  function init() {
    if (!loadConfig()) {
      renderError('Timeline configuration missing.');
      return;
    }
    fetchTimeline();
    bindAddJobEvents();
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
