<?php
/**
 * Planning - Add Job AJAX endpoint
 * Creates a new planning job.
 *
 * Required POST parameters:
 *   token          - Dolibarr CSRF token
 *   group_code     - 'forming' or 'trimming'
 *   fk_workstation - workstation ID
 *
 * Optional POST parameters:
 *   works_order_no  - works order number
 *   estimated_hours - estimated hours (default 1.0)
 *   date_start      - start datetime (YYYY-MM-DDTHH:MM or YYYY-MM-DD HH:MM:SS)
 *   date_end        - end datetime (YYYY-MM-DDTHH:MM or YYYY-MM-DD HH:MM:SS)
 *   notes           - notes text
 */

// Prevent direct output before headers
ob_start();

require '../../../main.inc.php';

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// Clear any buffered output
ob_end_clean();

/**
 * Output JSON response and exit.
 *
 * @param array $data Response data
 * @param int   $code HTTP status code
 */
function outputJson($data, $code = 200)
{
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

// Check CSRF token
$token = GETPOST('token', 'aZ09');
$sessionToken = isset($_SESSION['newtoken']) ? $_SESSION['newtoken'] : (isset($_SESSION['token']) ? $_SESSION['token'] : '');
if (empty($token) || $token !== $sessionToken) {
    outputJson(array('error' => 'Invalid or missing security token.'), 403);
}

// Check permissions - require addjob right
if (empty($user->rights->planning) || empty($user->rights->planning->addjob)) {
    outputJson(array('error' => 'Access denied. Missing planning addjob permission.'), 403);
}

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    outputJson(array('error' => 'Invalid request method.'), 405);
}

// Get and validate parameters
$group_code = GETPOST('group_code', 'aZ09');
$fk_workstation = GETPOST('fk_workstation', 'int');
$works_order_no = GETPOST('works_order_no', 'alphanohtml');
$estimated_hours = GETPOST('estimated_hours', 'alpha');
$date_start = GETPOST('date_start', 'alphanohtml');
$date_end = GETPOST('date_end', 'alphanohtml');
$notes = GETPOST('notes', 'restricthtml');

// Validate required fields
if (empty($group_code) || !in_array($group_code, array('forming', 'trimming'))) {
    outputJson(array('error' => 'Invalid group code. Must be forming or trimming.'), 400);
}

if (empty($fk_workstation) || $fk_workstation <= 0) {
    outputJson(array('error' => 'Please select a workstation.'), 400);
}

// Validate workstation exists
$sqlWsCheck = "SELECT rowid FROM " . MAIN_DB_PREFIX . "workstation_workstation WHERE rowid = " . (int) $fk_workstation;
$resWsCheck = $db->query($sqlWsCheck);
if (!$resWsCheck || !$db->fetch_object($resWsCheck)) {
    outputJson(array('error' => 'Selected workstation does not exist.'), 400);
}

// Parse estimated hours (default 1.0)
$estimated_hours = str_replace(',', '.', $estimated_hours);
$estimated_hours = (float) $estimated_hours;
if ($estimated_hours <= 0) {
    $estimated_hours = 1.0;
}

// Parse and format datetime fields (convert from datetime-local format)
$date_start_sql = null;
$date_end_sql = null;
if (!empty($date_start)) {
    // Handle both "YYYY-MM-DDTHH:MM" and "YYYY-MM-DD HH:MM:SS" formats
    $date_start = str_replace('T', ' ', $date_start);
    if (strlen($date_start) === 16) {
        $date_start .= ':00'; // Add seconds
    }
    $ts = strtotime($date_start);
    if ($ts !== false) {
        $date_start_sql = date('Y-m-d H:i:s', $ts);
    }
}
if (!empty($date_end)) {
    $date_end = str_replace('T', ' ', $date_end);
    if (strlen($date_end) === 16) {
        $date_end .= ':00';
    }
    $ts = strtotime($date_end);
    if ($ts !== false) {
        $date_end_sql = date('Y-m-d H:i:s', $ts);
    }
}

// Begin transaction
$db->begin();

try {
    // Get next sort_order for this workstation + group combination
    $sqlMax = "SELECT COALESCE(MAX(sort_order), 0) + 10 as next_sort
               FROM " . MAIN_DB_PREFIX . "planning_job
               WHERE fk_workstation = " . (int) $fk_workstation . "
               AND group_code = '" . $db->escape($group_code) . "'";
    $resMax = $db->query($sqlMax);
    $nextSort = 10;
    if ($resMax && ($objMax = $db->fetch_object($resMax))) {
        $nextSort = (int) $objMax->next_sort;
    }

    // Insert job
    $sql = "INSERT INTO " . MAIN_DB_PREFIX . "planning_job (
                fk_workstation,
                estimated_hours,
                works_order_no,
                notes,
                group_code,
                status,
                sort_order,
                date_start,
                date_end,
                entity,
                fk_user_creat,
                tms
            ) VALUES (
                " . (int) $fk_workstation . ",
                " . (float) $estimated_hours . ",
                '" . $db->escape($works_order_no) . "',
                '" . $db->escape($notes) . "',
                '" . $db->escape($group_code) . "',
                'planned',
                " . (int) $nextSort . ",
                " . ($date_start_sql ? "'" . $db->escape($date_start_sql) . "'" : "NULL") . ",
                " . ($date_end_sql ? "'" . $db->escape($date_end_sql) . "'" : "NULL") . ",
                " . (int) $conf->entity . ",
                " . (int) $user->id . ",
                NOW()
            )";

    $result = $db->query($sql);
    if (!$result) {
        throw new Exception('Database error: ' . $db->lasterror());
    }

    $newId = $db->last_insert_id(MAIN_DB_PREFIX . 'planning_job');

    $db->commit();

    outputJson(array(
        'success' => true,
        'message' => 'Job created successfully.',
        'job_id' => (int) $newId
    ), 200);

} catch (Exception $e) {
    $db->rollback();
    outputJson(array('error' => $e->getMessage()), 500);
}
