<?php
/* Dolibarr Scrap Management Class (patched version with MO origin link)
 * Compatible with Dolibarr 22.x
 */

require_once DOL_DOCUMENT_ROOT . '/core/class/commonobject.class.php';
require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT . '/product/stock/class/mouvementstock.class.php';
require_once DOL_DOCUMENT_ROOT . '/mrp/class/mo.class.php';

// --- Register Dolibarr object origins for Stock Movements (v22-compatible) ---
if (!isset($GLOBALS['objectsrc']) || !is_array($GLOBALS['objectsrc'])) $GLOBALS['objectsrc'] = array();
if (empty($GLOBALS['objectsrc']['mrp_mo'])) {
    $GLOBALS['objectsrc']['mrp_mo'] = '/mrp/class/mo.class.php';
}
if (empty($GLOBALS['objectsrc']['scrap'])) {
    $GLOBALS['objectsrc']['scrap'] = '/custom/scrap/class/scrap.class.php';
}

class Scrap extends CommonObject
{
    public $element = 'scrap';
    public $table_element = 'scrap';

    public $id;
    public $fk_product;
    public $fk_warehouse;
    public $qty;
    public $cost;
    public $reason;
    public $scrap_date;
    public $fk_user_scrap;
    public $fk_mo;
    public $error;
    public $errors = array();

    /**
     * Constructor
     */
    public function __construct($db)
    {
        $this->db = $db;
        $this->entity = $conf->entity ?? 1;
    }

    /**
     * Create new scrap record
     */
    public function create($user)
    {
        global $conf, $langs;

        $this->error = '';
        $this->errors = array();

        // --- Basic validation ---
        $this->fk_product   = (int)$this->fk_product;
        $this->fk_warehouse = (int)$this->fk_warehouse;
        $this->qty          = price2num($this->qty, 'MS');   // keep sign/decimals
        $this->cost         = price2num($this->cost, 'MU');  // unit cost
        $this->reason       = dol_string_nohtmltag(trim((string)$this->reason));

        if ($this->fk_product <= 0)  $this->errors[] = 'NoProduct';
        if ($this->fk_warehouse <= 0) $this->errors[] = 'NoWarehouse';
        if ($this->qty <= 0)         $this->errors[] = 'QtyMustBePositive';
        if ($this->cost < 0)         $this->errors[] = 'CostMustNotBeNegative';

        // Parse scrap_date from <input type="datetime-local">
        $ts = !empty($this->scrap_date) ? dol_stringtotime($this->scrap_date) : dol_now();
        if (empty($ts)) $ts = dol_now();
        $this->scrap_date = $this->db->idate($ts);

        if (!empty($this->errors)) {
            $this->error = implode(', ', $this->errors);
            return -1;
        }

        // --- Load product and ensure it's stockable (type 0 = product, 1 = service) ---
        $product = new Product($this->db);
        if ($product->fetch($this->fk_product) <= 0) {
            $this->error = 'ProductNotFound';
            return -1;
        }
        if ((int)$product->type !== 0) {
            $this->error = 'ProductIsNotStockable';
            return -2;
        }

        // --- Auto-link to a Manufacturing Order (MO) if not already provided ---
        if (empty($this->fk_mo)) {
            $sqlFindMo = "SELECT rowid
                          FROM " . MAIN_DB_PREFIX . "mrp_mo
                          WHERE fk_product = " . ((int)$this->fk_product) . "
                          AND status IN (1,2,3)        -- Validated / In progress / Produced
                          ORDER BY rowid DESC
                          LIMIT 1";
            $resFindMo = $this->db->query($sqlFindMo);
            if ($resFindMo && ($objFindMo = $this->db->fetch_object($resFindMo))) {
                $this->fk_mo = (int)$objFindMo->rowid;
            }
        }
        if (empty($this->fk_mo)) {
            $this->fk_mo = null;
        }

        // --- Warehouse ref just for nicer description text ---
        $warehouseRef = '';
        $sqlw = "SELECT ref FROM " . MAIN_DB_PREFIX . "entrepot WHERE rowid=" . (int)$this->fk_warehouse;
        $resw = $this->db->query($sqlw);
        if ($resw && ($objw = $this->db->fetch_object($resw))) {
            $warehouseRef = $objw->ref;
        }

        // Initials for trace ("MF", etc.)
        $userInitials = $this->_getUserInitials($user);

        // --- Start SQL transaction ---
        $this->db->begin();

        // --- Insert scrap row into llx_scrap ---
        $sql = "INSERT INTO " . MAIN_DB_PREFIX . "scrap(";
        $sql .= "fk_product, fk_warehouse, fk_mo, qty, cost, reason, scrap_date, fk_user_scrap, entity";
        $sql .= ") VALUES (";
        $sql .= (int)$this->fk_product . ",";
        $sql .= (int)$this->fk_warehouse . ",";
        $sql .= (!empty($this->fk_mo) ? (int)$this->fk_mo : "NULL") . ",";
        $sql .= "'" . $this->db->escape($this->qty) . "',";
        $sql .= "'" . $this->db->escape($this->cost) . "',";
        $sql .= (empty($this->reason) ? "NULL" : "'" . $this->db->escape($this->reason) . "'") . ",";
        $sql .= "'" . $this->db->escape($this->scrap_date) . "',";
        $sql .= (int)$user->id . ",";
        $sql .= (int)$conf->entity;
        $sql .= ")";

        if (!$this->db->query($sql)) {
            $this->error = $this->db->lasterror();
            $this->db->rollback();
            return -1;
        }

        $this->id = $this->db->last_insert_id(MAIN_DB_PREFIX . 'scrap');

        // --- Build human-readable stock movement description ---
        $dateStr   = dol_print_date($ts, 'dayhour');
        $qtyStr    = ($this->qty == (int)$this->qty) ? (string)(int)$this->qty : (string)$this->qty;
        $unitCost  = $this->cost;
        $totalCost = price2num($this->cost * $this->qty, 'MU');

        $desc = sprintf(
            "ScrapMovement: %s | Product: %s - %s | Warehouse: %s | Qty: %s | UnitCost: %s | TotalCost: %s | Reason: %s | By: %s",
            $dateStr,
            $product->ref,
            $product->label,
            ($warehouseRef !== '' ? $warehouseRef : $this->fk_warehouse),
            $qtyStr,
            price2num($unitCost, 'MU'),
            price2num($totalCost, 'MU'),
            ($this->reason !== '' ? $this->reason : 'N/A'),
            $userInitials
        );

        // --- Create the stock movement object ---
        $movement = new MouvementStock($this->db);

        // We'll fill these before calling livraison()
        $finalInventoryCode = '';
        $originRef = '';

        if (!empty($this->fk_mo)) {
            // Scrap is linked to a Manufacturing Order (MO)
            require_once DOL_DOCUMENT_ROOT . '/mrp/class/mo.class.php';
            $mo = new Mo($this->db);
            $moRef = '';
            if ($mo->fetch((int)$this->fk_mo) > 0) {
                $moRef = $mo->ref;
            }

            // Set movement origin for stock_mouvement "Origin" column
            $movement->fk_origin  = $mo->id;
            $movement->origintype = $mo->element;   // usually 'mrp_mo' or 'mo'
            $movement->setOrigin($mo->element, $mo->id);

            // This text appears in stock movement "Label"
            $movement->label = 'Scrap from ' . $moRef;

            // Try to re-use the same inventorycode that Dolibarr created
            // when consuming/producing for this same MO.
            // 1) Look for an existing movement with that MO as origin.
            $sqlInv = "SELECT inventorycode
                       FROM " . MAIN_DB_PREFIX . "stock_mouvement
                       WHERE fk_origin = " . ((int)$this->fk_mo) . "
                         AND (origintype = 'mo' OR origintype = 'mrp_mo')
                         AND inventorycode IS NOT NULL
                         AND inventorycode != ''
                       ORDER BY rowid ASC
                       LIMIT 1";
            $resInv = $this->db->query($sqlInv);
            if ($resInv && ($objInv = $this->db->fetch_object($resInv))) {
                $finalInventoryCode = $objInv->inventorycode;
            }

            // 2) Fallback: check mrp_production (some Dolibarr setups store it via mrp_production rows)
            if (empty($finalInventoryCode)) {
                $sqlInvProd = "SELECT sm.inventorycode
                               FROM " . MAIN_DB_PREFIX . "stock_mouvement AS sm
                               INNER JOIN " . MAIN_DB_PREFIX . "mrp_production AS mp
                                 ON sm.fk_origin = mp.rowid
                               WHERE mp.fk_mo = " . ((int)$this->fk_mo) . "
                                 AND sm.inventorycode IS NOT NULL
                                 AND sm.inventorycode != ''
                               ORDER BY sm.rowid ASC
                               LIMIT 1";
                $resInvProd = $this->db->query($sqlInvProd);
                if ($resInvProd && ($objInvProd = $this->db->fetch_object($resInvProd))) {
                    $finalInventoryCode = $objInvProd->inventorycode;
                }
            }

            // 3) If still empty, generate a new Dolibarr-style timestamp code
            if (empty($finalInventoryCode)) {
                $finalInventoryCode = dol_print_date(dol_now(), '%Y%m%d%H%M%S');
            }

            $movement->inventorycode = $finalInventoryCode;
            $originRef = $moRef; // for UI message
        } else {
            // Scrap is not linked to an MO
            $movement->fk_origin  = $this->id;
            $movement->origintype = 'scrap';
            $movement->setOrigin('scrap', $this->id);

            $movement->label = 'Scrap #' . $this->id;

            // Make our own code like SCRAP-YYYYMMDDHHMMSS
            $finalInventoryCode = 'SCRAP-' . dol_print_date(dol_now(), '%Y%m%d%H%M%S');
            $movement->inventorycode = $finalInventoryCode;

            $originRef = ''; // nothing to show as MO ref
        }

        // --- Actually move stock OUT (physical stock goes down) ---
        // livraison() returns rowid of llx_stock_mouvement if OK
        $resmove = $movement->livraison(
            $user,
            (int)$this->fk_product,
            (int)$this->fk_warehouse,
            (float)$this->qty,
            (float)$this->cost,
            $desc
        );

        if ($resmove <= 0) {
            $this->error = $movement->error ? $movement->error : 'StockMovementFailed';
            $this->db->rollback();
            return -3;
        }

        // --- Force inventorycode into llx_stock_mouvement so that:
        //     - Inv./Mov. code column shows it
        //     - It's consistent with MO consumption movements
        if (!empty($finalInventoryCode)) {
            $sqlUpdateInv = "UPDATE " . MAIN_DB_PREFIX . "stock_mouvement
                             SET inventorycode = '" . $this->db->escape($finalInventoryCode) . "'
                             WHERE rowid = " . ((int)$resmove);
            $this->db->query($sqlUpdateInv);
        }

        // --- Commit transaction ---
        $this->db->commit();

        // --- Prepare message for UI (same style used by delete()) ---
        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/product/stock/class/entrepot.class.php';
        require_once DOL_DOCUMENT_ROOT . '/mrp/class/mo.class.php';

        // --- Update Manufacturing Order qty_scrap ---
        if (!empty($this->fk_mo)) {
            $sqlUpdateMO = "UPDATE " . MAIN_DB_PREFIX . "mrp_mo
                            SET qty_scrap = COALESCE(qty_scrap, 0) + " . ((float)$this->qty) . "
                            WHERE rowid = " . ((int)$this->fk_mo);
            $this->db->query($sqlUpdateMO);
        }

        $productObj   = new Product($this->db);
        $warehouseObj = new Entrepot($this->db);
        $productObj->fetch($this->fk_product);
        $warehouseObj->fetch($this->fk_warehouse);

        // Show popup message with all details including total cost and inventory code
        $this->_showScrapMessage(
            'create',
            $productObj,
            $warehouseObj,
            $this->qty,
            $this->cost,
            $this->reason,
            $finalInventoryCode,
            $originRef
        );

        return (int)$this->id;
    }

    /**
     * Delete scrap record and restore stock + revert MO
     *
     * @param  User  $user  Current user
     * @param  int   $id    Scrap rowid
     * @return int|false    Movement ID or false on error
     */
    public function delete($user, $id)
    {
        global $langs;

        $id = (int) $id;
        if ($id <= 0) {
            $this->error = 'Invalid scrap ID';
            return -1;
        }

        dol_syslog(__METHOD__ . " Deleting scrap #{$id}", LOG_DEBUG);

        // --- Fetch scrap info (with fk_mo) ---
        $sql = "SELECT s.rowid, s.fk_product, s.fk_warehouse, s.fk_mo, s.qty, s.cost, s.reason,
                       p.ref AS product_ref, p.label AS product_label,
                       e.ref AS warehouse_ref
                FROM " . MAIN_DB_PREFIX . "scrap AS s
                LEFT JOIN " . MAIN_DB_PREFIX . "product AS p ON s.fk_product = p.rowid
                LEFT JOIN " . MAIN_DB_PREFIX . "entrepot AS e ON s.fk_warehouse = e.rowid
                WHERE s.rowid = " . $id;

        $res = $this->db->query($sql);
        if (!$res || $this->db->num_rows($res) == 0) {
            $this->error = 'ScrapNotFound';
            setEventMessages($langs->trans("ScrapNotFound"), null, 'errors');
            return -1;
        }
        $obj = $this->db->fetch_object($res);

        require_once DOL_DOCUMENT_ROOT . '/product/stock/class/mouvementstock.class.php';
        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/product/stock/class/entrepot.class.php';
        require_once DOL_DOCUMENT_ROOT . '/mrp/class/mo.class.php';

        $this->db->begin();

        $movement = new MouvementStock($this->db);
        $moRef = '';

        // --- Link movement to MO if available ---
        if (!empty($obj->fk_mo)) {
            $mo = new Mo($this->db);
            if ($mo->fetch((int)$obj->fk_mo) > 0) {
                $moRef = $mo->ref;
            }

            $movement->origintype = $mo->element ?: 'mrp_mo';
            $movement->fk_origin  = $mo->id;
            $movement->setOrigin($movement->origintype, $movement->fk_origin);
            $movement->label = 'Scrap Delete (from ' . ($moRef ?: 'MO #'.$mo->id) . ')';
        } else {
            $movement->origintype = 'scrap';
            $movement->fk_origin  = $id;
            $movement->setOrigin('scrap', $id);
            $movement->label = 'Scrap Delete #'.$id;
        }

        // --- Retrieve inventorycode from MO if exists ---
        $inventorycode = '';
        if (!empty($obj->fk_mo)) {
            $sqlInv = "SELECT inventorycode
                       FROM " . MAIN_DB_PREFIX . "stock_mouvement
                       WHERE fk_origin = " . ((int)$obj->fk_mo) . "
                         AND (origintype = 'mo' OR origintype = 'mrp_mo')
                         AND inventorycode IS NOT NULL
                         AND inventorycode != ''
                       ORDER BY rowid ASC
                       LIMIT 1";
            $resInv = $this->db->query($sqlInv);
            if ($resInv && ($inv = $this->db->fetch_object($resInv))) {
                $inventorycode = $inv->inventorycode;
            }
        }

        // fallback – generate new code if none found
        if (empty($inventorycode)) {
            $inventorycode = dol_print_date(dol_now(), '%Y%m%d%H%M%S');
        }

        $movement->inventorycode = $inventorycode;

        // --- Build movement description ---
        $userInitials = strtoupper(substr($user->firstname, 0, 1) . substr($user->lastname, 0, 1));
        $dateStr  = dol_print_date(dol_now(), 'dayhour');
        $qtyStr   = (fmod($obj->qty, 1) == 0) ? intval($obj->qty) : number_format($obj->qty, 2);
        $totalCost = price2num($obj->cost * $obj->qty, 'MU');

        $desc = sprintf(
            "ScrapDeletedAndStockRestored: %s | Product: %s - %s | Warehouse: %s | Qty Restored: %s | UnitCost: %s | TotalCost: %s | Reason: %s | By: %s",
            $dateStr,
            $obj->product_ref,
            $obj->product_label,
            $obj->warehouse_ref,
            $qtyStr,
            price2num($obj->cost, 'MU'),
            $totalCost,
            (!empty($obj->reason) ? $obj->reason : 'N/A'),
            $userInitials
        );

        // --- Stock reception (reverse movement) ---
        $resmove = $movement->reception(
            $user,
            (int)$obj->fk_product,
            (int)$obj->fk_warehouse,
            abs((float)$obj->qty),
            (float)$obj->cost,
            $desc
        );

        // force inventorycode update (Dolibarr sometimes skips it)
        if ($resmove > 0 && !empty($inventorycode)) {
            $sqlInvUpd = "UPDATE " . MAIN_DB_PREFIX . "stock_mouvement
                          SET inventorycode = '" . $this->db->escape($inventorycode) . "'
                          WHERE rowid = " . ((int)$resmove);
            $this->db->query($sqlInvUpd);
        }

        if ($resmove <= 0) {
            $this->db->rollback();
            $this->error = 'StockRestoreFailed';
            setEventMessages($langs->trans("StockRestoreFailed"), null, 'errors');
            return -1;
        }

        // --- Delete scrap record ---
        $this->db->query("DELETE FROM " . MAIN_DB_PREFIX . "scrap WHERE rowid = " . $id);

        // --- Revert MO scrap quantity if linked ---
        if (!empty($obj->fk_mo)) {
            $sqlMo = "SELECT rowid, qty_scrap FROM " . MAIN_DB_PREFIX . "mrp_mo WHERE rowid = " . ((int)$obj->fk_mo);
            $resMo = $this->db->query($sqlMo);
            if ($resMo && ($moObj = $this->db->fetch_object($resMo))) {
                $newScrapQty = max(0, $moObj->qty_scrap - abs((float)$obj->qty));
                $this->db->query(
                    "UPDATE " . MAIN_DB_PREFIX . "mrp_mo
                     SET qty_scrap = " . price2num($newScrapQty) . "
                     WHERE rowid = " . ((int)$moObj->rowid)
                );
            }
        }

        $this->db->commit();

        // --- Build Product & Warehouse objects for the message ---
        $productObj = new Product($this->db);
        $warehouseObj = new Entrepot($this->db);
        $productObj->fetch($obj->fk_product);
        $warehouseObj->fetch($obj->fk_warehouse);

        // --- Show unified success message (uses _showScrapMessage) ---
        $this->_showScrapMessage(
            'delete',
            $productObj,
            $warehouseObj,
            $obj->qty,
            $obj->cost,
            $obj->reason,
            $inventorycode,
            $moRef
        );

        return $resmove;
    }



    /**
     * Display standardized success message after creating or deleting scrap
     *
     * @param string $action      'create' or 'delete'
     * @param object $productObj  Product object
     * @param object $warehouseObj Warehouse object
     * @param float  $qty
     * @param float  $cost
     * @param string $reason
     * @param string $inventorycode
     * @param string $originRef
     */
    private function _showScrapMessage(
        $mode,
        $productObj,
        $warehouseObj,
        $qty,
        $unitCost,
        $reason,
        $inventorycode,
        $originRef
    ) {
        global $langs;

        $totalCost = price2num(((float)$qty) * ((float)$unitCost), 'MU');
        $qtyStr = (fmod($qty, 1.0) == 0.0) ? (string)((int)$qty) : (string)$qty;

        $title = ($mode === 'delete')
            ? $langs->trans("ScrapDeletedAndStockRestored")
            : $langs->trans("ScrapCreatedAndStockUpdated");

        $msg  = $title . "<br>";
        $msg .= $langs->trans("Product") . ": " . dol_escape_htmltag($productObj->ref . " - " . $productObj->label) . "<br>";
        $msg .= $langs->trans("Warehouse") . ": " . dol_escape_htmltag($warehouseObj->ref) . "<br>";
        $msg .= $langs->trans("Qty") . ": " . $qtyStr . "<br>";
        $msg .= $langs->trans("Unit Cost") . ": " . price($unitCost, 0, $langs, 0, 0, -1) . "<br>";
        $msg .= $langs->trans("Total Cost") . ": " . price($totalCost, 0, $langs, 0, 0, -1) . "<br>";
        $msg .= $langs->trans("Reason") . ": "
              . (!empty($reason) ? dol_escape_htmltag($reason) : $langs->trans("NoReasonProvided")) . "<br>";

        if (!empty($inventorycode)) {
            $msg .= "Inv./Mov. code: <strong>" . dol_escape_htmltag($inventorycode) . "</strong><br>";
        }

        if (!empty($originRef)) {
            $msg .= "Origin: " . dol_escape_htmltag($originRef) . "<br>";
        }

        setEventMessages($msg, null, 'mesgs');
    }

    /**
     * Fetch scrap record
     */
    public function fetch($id)
    {
        $sql = "SELECT * FROM " . MAIN_DB_PREFIX . "scrap WHERE rowid = " . ((int)$id);
        $res = $this->db->query($sql);
        if ($res && $obj = $this->db->fetch_object($res)) {
            foreach ($obj as $key => $val) {
                $this->$key = $val;
            }
            return 1;
        }
        return -1;
    }

    /**
     * Compute user initials like "AS"
     */
    private function _getUserInitials($user)
    {
        $i1 = '';
        $i2 = '';
        if (!empty($user->firstname)) $i1 = mb_substr($user->firstname, 0, 1, 'UTF-8');
        if (!empty($user->lastname))  $i2 = mb_substr($user->lastname, 0, 1, 'UTF-8');
        $ini = strtoupper($i1.$i2);
        if ($ini === '' && !empty($user->login)) {
            $ini = strtoupper(mb_substr($user->login, 0, 2, 'UTF-8'));
        }
        return $ini ?: 'NA';
    }
}
