/*
 * ProductionRules - Sales Order rules confirmation modal
 * Main screen must NOT validate or generate anything.
 * Generation happens ONLY after clicking "CONFIRM & GENERATE" in the modal.
 */

(function () {
  'use strict';

  function qs(sel, root) {
    return (root || document).querySelector(sel);
  }

  function qsa(sel, root) {
    return Array.prototype.slice.call((root || document).querySelectorAll(sel));
  }

  function esc(s) {
    return (s == null ? '' : String(s))
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function openModal() {
    var modal = qs('#prSoConfirmModal');
    if (!modal) return;
    modal.classList.add('open');
  }

  function closeModal() {
    var modal = qs('#prSoConfirmModal');
    if (!modal) return;
    modal.classList.remove('open');
  }

  function readComponentsFromTable() {
    // Reads components from the right-hand table (if present)
    var rows = qsa('table.noborder tr.oddeven');
    var out = [];
    rows.forEach(function (tr) {
      var tds = tr.querySelectorAll('td');
      if (!tds || tds.length < 2) return;
      var ref = (tds[0].textContent || '').trim();
      var qty = parseFloat((tds[1].textContent || '').trim().replace(',', '.'));
      if (!ref) return;
      if (!isFinite(qty) || qty <= 0) qty = 1;
      out.push({ ref: ref, qty: qty });
    });
    return out;
  }

  function buildPreview() {
    var form = qs('#prSoMainForm');
    if (!form) return null;

    var socidSel = qs('select[name="socid"]', form);
    var customerId = socidSel ? (socidSel.value || '') : '';
    var customerName = '';
    if (socidSel && socidSel.selectedOptions && socidSel.selectedOptions[0]) {
      customerName = (socidSel.selectedOptions[0].textContent || '').trim();
    }

    var sellRefEl = qs('input[name="sell_ref"]', form);
    var sellQtyEl = qs('input[name="sell_qty"]', form);
    var sellRef = ((sellRefEl && sellRefEl.value) ? sellRefEl.value : '').trim();
    var sellQtyRaw = ((sellQtyEl && sellQtyEl.value) ? sellQtyEl.value : '').trim();
    var sellQty = parseFloat(sellQtyRaw.replace(',', '.'));
    if (!isFinite(sellQty) || sellQty <= 0) sellQty = 1;

    var jobEl = qs('input[name="job_number"]', form);
    var job = ((jobEl && jobEl.value) ? jobEl.value : '').trim();

    var prSel = qs('select[name="prod_ruleset_id"]', form);
    var prodRulesetId = prSel ? (prSel.value || '') : '';
    var prodRulesetLabel = '';
    if (prSel && prSel.selectedOptions && prSel.selectedOptions[0]) {
      prodRulesetLabel = (prSel.selectedOptions[0].textContent || '').trim();
    }

    var materialEl = qs('input[name="material"]', form);
    var material = ((materialEl && materialEl.value) ? materialEl.value : '').trim();

    var comps = readComponentsFromTable();

    // Render preview blocks
    var soBox = qs('#prSoPreviewSO');
    var compBox = qs('#prSoPreviewComponents');
    var prBox = qs('#prSoPreviewProd');

    if (soBox) {
      var html = '';
      html += '<div><b>Customer:</b> ' + esc(customerName || '') + '</div>';
      html += '<div><b>Sell:</b> ' + esc(sellRef) + '</div>';
      html += '<div><b>Qty:</b> ' + esc(sellQty) + '</div>';
      if (job) html += '<div><b>Job:</b> ' + esc(job) + '</div>';
      soBox.innerHTML = html;
    }

    if (compBox) {
      var lines = [];
      lines.push('Components:');
      comps.forEach(function (c) {
        var total = c.qty * sellQty;
        lines.push('- ' + c.ref + ' x ' + c.qty + '   (SO total: ' + total + ')');
      });
      compBox.textContent = lines.join('\n');
    }

    if (prBox) {
      var html2 = '';
      html2 += '<div><b>Production RuleSet:</b> ' + esc(prodRulesetLabel) + '</div>';
      if (material) html2 += '<div><b>Material packs:</b> ' + esc(material) + '</div>';
      prBox.innerHTML = html2;
    }

    return {
      customerId: customerId,
      sellRef: sellRef,
      sellQty: sellQty,
      job: job,
      prodRulesetId: prodRulesetId,
      material: material,
      components: comps
    };
  }

  function wire() {
    var overlay = qs('#prSoConfirmModal');
    if (overlay) {
      overlay.addEventListener('click', function (e) {
        // Click on the dark overlay (not inside the card) closes
        if (e.target === overlay) closeModal();
      });
    }

    var openBtn = qs('#prSoOpenConfirm');
    if (openBtn) {
      openBtn.addEventListener('click', function () {
        buildPreview();
        openModal();
      });
    }

    qsa('[data-pr-close="1"]').forEach(function (el) {
      el.addEventListener('click', function () {
        closeModal();
      });
    });

    // ESC closes
    document.addEventListener('keydown', function (e) {
      if (e.key === 'Escape') closeModal();
    });

    var confirmBtn = qs('#prSoConfirmGenerate');
    if (confirmBtn) {
      confirmBtn.addEventListener('click', function () {
        var data = buildPreview();
        var f = qs('#prSoConfirmForm');
        if (!f || !data) return;

        // Copy values to hidden form and submit
        var el;
        el = qs('#prSoF_socid'); if (el) el.value = data.customerId;
        el = qs('#prSoF_sell_ref'); if (el) el.value = data.sellRef;
        el = qs('#prSoF_sell_qty'); if (el) el.value = String(data.sellQty);
        el = qs('#prSoF_prod_ruleset_id'); if (el) el.value = data.prodRulesetId;
        el = qs('#prSoF_material'); if (el) el.value = data.material;
        el = qs('#prSoF_job_number'); if (el) el.value = data.job;

        // Submit (server-side validation + generation)
        f.submit();
      });
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', wire);
  } else {
    wire();
  }
})();
