<?php

/**
 * Class ProdRules
 * Manage production rules and simple BOM generation
 */

require_once DOL_DOCUMENT_ROOT . '/core/class/commonobject.class.php';

class ProdRules extends CommonObject
{
    public $lastDownstreamJobMap = array();

    public $element = 'prod_rules';
    public $table_element = 'prod_rules';

    /** @var DoliDB */
    public $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Get list of rules with optional filter
     *
     * @param array $filter
     * @return array
     */
    public function fetchAll($filter = array())
    {
        global $user;

        $sql = "SELECT * FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " AND fk_user_creat = " . (int) $user->id;   // ⬅️ TYLKO swoje reguły

        if (!empty($filter['from_ref'])) {
            $sql .= " AND from_ref = '" . $this->db->escape($filter['from_ref']) . "'";
        }
        if (!empty($filter['to_ref'])) {
            $sql .= " AND to_ref = '" . $this->db->escape($filter['to_ref']) . "'";
        }
        if (!empty($filter['fk_ruleset'])) {
            $sql .= " AND fk_ruleset = " . (int) $filter['fk_ruleset'];
        }
        if (isset($filter['active'])) {
            $sql .= " AND active = " . (int) $filter['active'];
        }

        $sql .= " ORDER BY step_order, rowid";

        $res = $this->db->query($sql);
        if (!$res) {
            return array();
        }

        $out = array();
        while ($obj = $this->db->fetch_object($res)) {
            $out[] = $obj;
        }

        return $out;
    }
    /**
     * Insert one rule
     *
     * @param User  $user
     * @param array $data
     * @return int  new id or -1 on error
     */
public function createRule($user, $data)
{
    global $conf;

    $now = $this->db->idate(dol_now());

    $sql = "INSERT INTO " . MAIN_DB_PREFIX . "prod_rules(";
    $sql .= "from_ref, operation, to_ref, fk_ruleset, qty_per_from, workstation, step_order, ";
    $sql .= "is_final, is_raw, active, note, datec, fk_user_creat, entity";
    $sql .= ") VALUES (";

    // from_ref
    $sql .= "'" . $this->db->escape($data['from_ref']) . "',";

    // operation
    $sql .= "'" . $this->db->escape($data['operation']) . "',";

    // to_ref
    $sql .= "'" . $this->db->escape($data['to_ref']) . "',";

    // fk_ruleset
    if (!empty($data['fk_ruleset'])) {
        $sql .= (int) $data['fk_ruleset'] . ",";
    } else {
        $sql .= "NULL,";
    }

    // qty_per_from
    $qty = (isset($data['qty_per_from']) ? (float) $data['qty_per_from'] : 1.0);
    $sql .= $qty . ",";

    // workstation
    if (!empty($data['workstation'])) {
        $sql .= "'" . $this->db->escape($data['workstation']) . "',";
    } else {
        $sql .= "NULL,";
    }

    // step_order
    $sql .= (int) (isset($data['step_order']) ? $data['step_order'] : 10) . ",";

    // is_final
    $sql .= (int) (!empty($data['is_final']) ? 1 : 0) . ",";

    // is_raw
    $sql .= (int) (!empty($data['is_raw']) ? 1 : 0) . ",";

    // active
    $sql .= "1,";

    // note
    if (!empty($data['note'])) {
        $sql .= "'" . $this->db->escape($data['note']) . "',";
    } else {
        $sql .= "NULL,";
    }

    // datec
    $sql .= "'" . $this->db->escape($now) . "',";

    // fk_user_creat
    $sql .= (int) $user->id . ",";

    // entity
    $sql .= (int) $conf->entity;

    $sql .= ")";

    $res = $this->db->query($sql);
    if (!$res) {
        $this->error = $this->db->lasterror();
        return -1;
    }

    return $this->db->last_insert_id(MAIN_DB_PREFIX . "prod_rules");
}

    /**
     * Compute raw requirements for given target product and quantity
     * (simplified version)
     *
     * @param string $target_ref
     * @param float  $qty_target
     * @return array ref => qty
     */
    public function computeRequirements($target_ref, $qty_target)
    {
        global $user;

        $need  = array();
        $stack = array();

        $stack[] = array($target_ref, $qty_target);

        while (!empty($stack)) {
            list($ref, $qty) = array_pop($stack);

            // 1) Ustalamy, czy ref jest surowcem wg reguł TEGO użytkownika
            $sql = "SELECT is_raw FROM " . MAIN_DB_PREFIX . "prod_rules";
            $sql .= " WHERE from_ref = '" . $this->db->escape($ref) . "'";
            $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
            $sql .= " AND fk_user_creat = " . (int) $user->id;
            $sql .= " ORDER BY rowid DESC LIMIT 1";

            $res = $this->db->query($sql);
            $is_raw = 0;
            if ($res && ($obj = $this->db->fetch_object($res))) {
                $is_raw = (int) $obj->is_raw;
            }

            if ($is_raw) {
                if (!isset($need[$ref])) {
                    $need[$ref] = 0;
                }
                $need[$ref] += $qty;
                continue;
            }

            // 2) Szukamy reguł, które tworzą ten ref jako to_ref – też tylko dla tego usera
            $sql = "SELECT * FROM " . MAIN_DB_PREFIX . "prod_rules";
            $sql .= " WHERE to_ref = '" . $this->db->escape($ref) . "'";
            $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
            $sql .= " AND fk_user_creat = " . (int) $user->id;

            $res = $this->db->query($sql);
            if (!$res || $this->db->num_rows($res) == 0) {
                // Treat as raw if no rules for this user
                if (!isset($need[$ref])) {
                    $need[$ref] = 0;
                }
                $need[$ref] += $qty;
                continue;
            }

            while ($r = $this->db->fetch_object($res)) {
                $from_ref     = $r->from_ref;
                $qty_per_from = (float) $r->qty_per_from;
                if ($qty_per_from == 0) {
                    $qty_per_from = 1.0;
                }

                $needed_from = $qty / $qty_per_from;
                $stack[] = array($from_ref, $needed_from);
            }
        }

        return $need;
    }

    /**
     * Generate a simple BOM in Dolibarr for a given FG
     * (FG -> list of raw requirements)
     *
     * @param User   $user
     * @param string $fg_ref
     * @param float  $qty_fg
     * @return int   bom id or -1 on error
     */
    public function generateBomForProduct($user, $fg_ref, $qty_fg = 1.0, $job_number = null, $fk_ruleset = null, $raw_ref = null)
    {
        global $conf, $langs;
        // UWAGA: tutaj $user jest parametrem, ale zostawmy go – nie trzeba global

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        // 1) Pobierz produkt, dla którego robimy BOM
        $prod = new Product($this->db);
        $res = $prod->fetch('', $fg_ref);
        if ($res <= 0) {
            $this->error = 'Product with ref ' . $fg_ref . ' not found';
            return -1;
        }
    
        // 2) ZBUDUJ WYMAGANIA JEDNEGO POZIOMU:
        //    wszystkie reguły, gdzie to_ref = nasz produkt, TYLKO bieżący user
        $requirements = array();

        $sql  = "SELECT from_ref, qty_per_from, step_order";
        $sql .= " FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE to_ref = '" . $this->db->escape($fg_ref) . "'";
        $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " AND fk_user_creat = " . (int) $user->id;   // ⬅️ tu dopisujemy

        if (!empty($fk_ruleset)) {
            $sql .= " AND fk_ruleset = " . (int) $fk_ruleset;
        }

        $sql .= " ORDER BY step_order, rowid";

        $resql = $this->db->query($sql);
        if (! $resql) {
            $this->error = 'Error selecting rules: ' . $this->db->lasterror();
            return -1;
        }

        while ($obj = $this->db->fetch_object($resql)) {
            $ref = $obj->from_ref;

            if (!isset($requirements[$ref])) {
                $requirements[$ref] = array(
                    'qty'        => 0,
                    'step_order' => (int) $obj->step_order
                );
            }

            // Ilość komponentu = ilość FG w BOM * ilość z reguły
            $requirements[$ref]['qty'] += ((float) $qty_fg * (float) $obj->qty_per_from);

            // step_order – bierzemy najmniejszy krok, jeśli jest więcej niż jedna reguła
            if ((int) $obj->step_order < $requirements[$ref]['step_order']) {
                $requirements[$ref]['step_order'] = (int) $obj->step_order;
            }
        }

        
        // Dodaj RAW materiał jako komponent wejściowy (1:1 do ilości FG)
        if (!empty($raw_ref)) {
            if (!isset($requirements[$raw_ref])) {
                $requirements[$raw_ref] = array(
                    'qty'        => 0,
                    'step_order' => 0
                );
            }
            // RAW zużywany 1:1 względem FG
            $requirements[$raw_ref]['qty'] += (float) $qty_fg;

            // Upewnij się, że raw ma najniższy krok (na początku listy)
            if ($requirements[$raw_ref]['step_order'] > 0) {
                $requirements[$raw_ref]['step_order'] = 0;
            }
        }

// 3) Utwórz BOM (nagłówek)
        $bom = new Bom($this->db);

        if (!empty($job_number)) {
            $bom->ref   = 'BOM-' . $fg_ref . '-' . $job_number;
            $bom->label = 'Auto BOM for ' . $fg_ref . ' (Job ' . $job_number . ')';
        } else {
            $bom->ref   = 'BOM-' . $fg_ref;
            $bom->label = 'Auto BOM for ' . $fg_ref;
        }

        $bom->fk_product = $prod->id;
        $bom->qty        = $qty_fg;
        $bom->status     = 0;
        $bom->entity     = $conf->entity;

        $res = $bom->create($user);
        if ($res <= 0) {
            $this->error = $bom->error;
            return -1;
        }

        // 4) Dodaj linie BOM na podstawie $requirements
        foreach ($requirements as $ref => $info) {
            $lineqty    = $info['qty'];
            $step_order = $info['step_order'];

            // Znajdź produkt komponentu
            $p = new Product($this->db);
            if ($p->fetch('', $ref) <= 0) {
                // brak produktu – log i pomijamy
                dol_syslog("ProductionRules::generateBomForProduct missing product ".$ref, LOG_WARNING);
                continue;
            }

            $line = new BomLine($this->db);
            $line->fk_bom     = $bom->id;
            $line->fk_product = $p->id;
            $line->qty        = $lineqty;
            $line->position   = $step_order;

            $resline = $line->create($user);
            if ($resline <= 0) {
                $this->error = 'Error creating BOM line for '.$ref.' : '.$line->error;
                return -1;
            }
        }

        // 5) Zatwierdź BOM
        $bom->status = 1;
        $bom->update($user);

        return $bom->id;
    }

    /**
     * Generate BOM for whole RuleSet (SF product) using dynamic material.
     *
     * @param User   $user
     * @param int    $fk_ruleset   ID of RuleSet
     * @param string $sf_ref       Product ref for SF (RuleSet ref)
     * @param float  $qty_sf       Quantity of SF
     * @param string $job_number   Optional job reference
     * @param string $raw_ref      Material (dynamic sheet) ref
     * @return int                 BOM id (>0) or <0 on error
     */
    
    /**
     * Generate BOM for whole RuleSet (SF product) using dynamic material.
     *
     * @param User   $user
     * @param int    $fk_ruleset   ID of RuleSet
     * @param string $sf_ref       Product ref for SF (RuleSet ref)
     * @param float  $qty_sf       Quantity of SF
     * @param string $job_number   Optional job reference
     * @param string $raw_ref      Material (dynamic sheet) ref
     * @return int                 BOM id (>0) or <0 on error
     */
    public function generateBomForRuleset($user, $fk_ruleset, $sf_ref, $qty_sf = 1.0, $job_number = null, $raw_ref = null)
    {
        global $conf, $langs;

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        // 1) Pobierz produkt SF, dla którego robimy BOM
        $prod = new Product($this->db);
        $res = $prod->fetch('', $sf_ref);
        if ($res <= 0) {
            $this->error = 'Product with ref ' . $sf_ref . ' not found';
            return -1;
        }

        if (empty($raw_ref)) {
            $this->error = 'Material ref is empty';
            return -1;
        }

        //         // Material: lista pack numberów po przecinku (tylko cyfry). Pack number musi być unikalny.
        // Zasada: 1 pack = 1 blacha = 1 SF
        $packs = array();
        $seen = array();
        $rawTokens = preg_split('/\s*,\s*/', trim($raw_ref));
        foreach ($rawTokens as $tok) {
            if ($tok === '') continue;

            if (!preg_match('/^\d+$/', $tok)) {
                $this->error = 'Material must be numeric pack numbers. Bad token: ' . $tok;
                return -1;
            }
            if (isset($seen[$tok])) {
                $this->error = 'Duplicate material pack number: ' . $tok;
                return -1;
            }
            $seen[$tok] = true;
            $packs[] = $tok;
        }

        if (empty($packs)) {
            $this->error = 'Material list is empty';
            return -1;
        }

        // Ilość SF wynika z ilości packów (1 blacha = 1 SF)
        $qty_sf = (float) count($packs);


        // 2) Utwórz nagłówek BOM
        $bom = new Bom($this->db);

        if (!empty($job_number)) {
            // Używamy wzoru BOM-{JOB}
            $bom->ref = 'BOM-' . $job_number;
        } else {
            // domyślna ref: SF + data
            $bom->ref = 'BOM-' . $sf_ref . '-' . dol_print_date(dol_now(), '%Y%m%d-%H%M%S');
        }

        if (!empty($job_number)) {
            $bom->label = 'Auto BOM for ' . $sf_ref . ' (Job ' . $job_number . ')';
        } else {
            $bom->label = 'Auto BOM for ' . $sf_ref;
        }
        $bom->quantity   = (float) $qty_sf;
        $bom->fk_product = $prod->id;
        $bom->fk_user    = $user->id;
        $bom->entity     = $conf->entity;

        // Zbuduj opis routingu (lista operacji po step_order) jako notatkę
        $sql  = "SELECT DISTINCT step_order, operation";
        $sql .= " FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE fk_ruleset = " . ((int) $fk_ruleset);
        $sql .= " AND active = 1";
        $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " ORDER BY step_order ASC";

        $resql = $this->db->query($sql);
        if ($resql) {
            $lines = array();
            while ($obj = $this->db->fetch_object($resql)) {
                $step = (int) $obj->step_order;
                $op   = trim($obj->operation);
                if ($op === '') continue;
                $lines[] = sprintf('%d %s', $step, $op);
            }
            // Preview / audit note (zapisywane do BOM)
            $previewLines = array();
            $previewLines[] = 'Generated by ProductionRules';
            if (!empty($job_number)) $previewLines[] = 'Job: ' . $job_number;
            $previewLines[] = 'Qty SF: ' . (int) $qty_sf;
            $previewLines[] = '';
            $previewLines[] = 'Material packs:';
            foreach ($packs as $p) {
                $previewLines[] = '- ' . $p;
            }
            $previewLines[] = '';
            $previewLines[] = $langs->transnoentitiesnoconv('Routing') . ':';
            foreach ($lines as $l) $previewLines[] = $l;

            $bom->note_public = implode("\n", $previewLines);
        }

        $resbom = $bom->create($user);
        if ($resbom <= 0) {
            $this->error = 'Error creating BOM: ' . $bom->error;
            return -1;
        }

        // 3) Dodaj linie BOM dla materiałów (wiele packów)
        $pos = 10;
        foreach ($packs as $packref) {
            $prodMat = new Product($this->db);
            $res2 = $prodMat->fetch('', $packref);
            if ($res2 <= 0) {
                $this->error = 'Material product with ref ' . $packref . ' not found';
                return -1;
            }

            $line = new BomLine($this->db);
            $line->fk_bom      = $bom->id;
            $line->fk_product  = $prodMat->id;

            // 1 pack = 1 blacha
            $line->qty         = 1.0;

            $line->position    = $pos;
            $pos += 10;
            $line->fk_unit     = $prodMat->fk_unit;
            $line->desc        = $langs->transnoentitiesnoconv('Material pack for SF') . ' ' . $sf_ref;

            $resline = $line->create($user);
            if ($resline <= 0) {
                $this->error = 'Error creating BOM line for material ' . $packref . ' : ' . $line->error;
                return -1;
            }
        }


        // 4) Zatwierdź BOM
        $bom->status = 1;
        $bom->update($user);

        
        // 5) Create downstream BOMs (SF->FG roots get new job numbers, FG->FG keep branch job)
        $this->generateDownstreamBoms($user, $fk_ruleset, $sf_ref, $job_number, $qty_sf);

        // Append branch job mapping to SF BOM note_public (if any)
        if (!empty($this->lastDownstreamJobMap)) {
            $bom->note_public .= "\n\nBranch jobs:";
            foreach ($this->lastDownstreamJobMap as $fgref => $job) {
                $bom->note_public .= "\n- " . $fgref . " -> " . (int) $job;
            }
            $bom->update($user);
        }


return $bom->id;
    }


    /**
     * Generate downstream BOMs (SF->FG and FG->FG) for a ruleset.
     * - Assign new job numbers ONLY for root edges SF->FG (sorted by to_ref).
     * - Reuse the assigned job number for the whole branch (FG->FG...).
     */
    public function generateDownstreamBoms($user, $fk_ruleset, $sf_ref, $base_job_number, $qty_sf = 1.0)
    {
        global $conf, $langs;

        if (empty($base_job_number) || !preg_match('/^\d+$/', (string) $base_job_number)) {
            // No numeric job base -> nothing to do (keep safe)
            return 0;
        }

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        $baseJob = (int) $base_job_number;
        $nextJob = $baseJob + 1;

        // Fetch all rules for this ruleset (active)
        $sql  = "SELECT from_ref, to_ref, qty_per_from, step_order, operation, workstation";
        $sql .= " FROM " . MAIN_DB_PREFIX . "prod_rules";
        $sql .= " WHERE fk_ruleset = " . ((int) $fk_ruleset);
        $sql .= " AND active = 1";
        $sql .= " AND entity IN (" . getEntity('prod_rules') . ")";
        $sql .= " ORDER BY rowid ASC";

        $resql = $this->db->query($sql);
        if (!$resql) {
            $this->error = $this->db->lasterror();
            return -1;
        }

        $edges = array();      // from => list of edges (to, qty)
        $routing = array();    // ref => list of routing lines "step op ws"
        $roots = array();      // SF->FG to_ref list

        while ($obj = $this->db->fetch_object($resql)) {
            $from = trim($obj->from_ref);
            $to   = trim($obj->to_ref);
            if ($from === '' || $to === '') continue;

            if ($from === $to) {
                // routing line for product $from
                $step = (int) $obj->step_order;
                $op = trim($obj->operation);
                $ws = trim($obj->workstation);
                if ($op !== '') {
                    if (!isset($routing[$from])) $routing[$from] = array();
                    $line = (string) $step . ' ' . $op;
                    if ($ws !== '') $line .= ' ' . $ws;
                    $routing[$from][] = array('step' => $step, 'txt' => $line);
                }
                continue;
            }

            // dependency edge
            if (!isset($edges[$from])) $edges[$from] = array();
            $edges[$from][] = array('to' => $to, 'qty' => (float) $obj->qty_per_from);

            // Root split: SF->FG only
            if ($from === $sf_ref && preg_match('/^FG/i', $to)) {
                $roots[$to] = true;
            }
        }

        // Sort routing lines by step_order
        foreach ($routing as $ref => $arr) {
            usort($arr, function($a, $b) {
                return $a['step'] <=> $b['step'];
            });
            $routing[$ref] = array_map(function($x){ return $x['txt']; }, $arr);
        }

        // Root FG list (Option A): alphabetical by to_ref
        $rootFgs = array_keys($roots);
        sort($rootFgs, SORT_STRING);

        // Assign job per root FG, then traverse downstream with same job
        $assignedJob = array(); // productRef => job
        foreach ($rootFgs as $fgref) {
            $assignedJob[$fgref] = $nextJob;
            $nextJob++;
        }

        // Ensure SF product object for component usage
        $sfProd = new Product($this->db);
        if ($sfProd->fetch('', $sf_ref) <= 0) {
            $this->error = 'Product with ref ' . $sf_ref . ' not found for downstream BOM generation';
            return -1;
        }

        $created = 0;
        $visited = array();

        foreach ($rootFgs as $fgref) {
            $job = $assignedJob[$fgref];

            // Create BOM for root FG: component is SF with qty = qty_per_from from SF->FG edge
            $qtyComp = 1.0;
            if (isset($edges[$sf_ref])) {
                foreach ($edges[$sf_ref] as $e) {
                    if ($e['to'] === $fgref) { $qtyComp = (float) $e['qty']; break; }
                }
            }

            $res = $this->createOrUpdateSimpleBom($user, $fgref, $job, $sfProd, $qtyComp, isset($routing[$fgref]) ? $routing[$fgref] : array(), $qty_sf);
            if ($res < 0) return -1;
            if ($res > 0) $created++;

            // Traverse FG->FG downstream
            $stack = array($fgref);
            while (!empty($stack)) {
                $current = array_pop($stack);
                if (isset($visited[$fgref.'>'.$current])) continue;
                $visited[$fgref.'>'.$current] = true;

                if (empty($edges[$current])) continue;

                foreach ($edges[$current] as $e) {
                    $child = $e['to'];
                    $qtyChild = (float) $e['qty'];

                    // Only follow FG->FG chains (as per your process)
                    if (!preg_match('/^FG/i', $child)) continue;

                    // child uses same job as branch root
                    $res2 = $this->createOrUpdateSimpleBom($user, $child, $job, $this->fetchProductByRef($current), $qtyChild, isset($routing[$child]) ? $routing[$child] : array(), $qty_sf);
                    if ($res2 < 0) return -1;
                    if ($res2 > 0) $created++;

                    $stack[] = $child;
                }
            }
        }

        
        // expose mapping for caller (SF->FG jobs)
        $this->lastDownstreamJobMap = $assignedJob;

return $created;
    }

    /**
     * Fetch product by ref (returns Product or null)
     */
    private function fetchProductByRef($ref)
    {
        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        $p = new Product($this->db);
        if ($p->fetch('', $ref) > 0) return $p;
        return null;
    }

    /**
     * Create a BOM for productRef if not already existing for that job label.
     * Component is componentProd with qty.
     * Returns 1 if created, 0 if skipped (exists), -1 on error.
     */
    private function createOrUpdateSimpleBom($user, $productRef, $job, $componentProd, $componentQty, $routingLines, $qty_sf)
    {
        global $conf;

        if (empty($componentProd) || empty($componentProd->id)) {
            $this->error = 'Component product not found';
            return -1;
        }

        require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bom.class.php';
        require_once DOL_DOCUMENT_ROOT . '/bom/class/bomline.class.php';

        $prod = new Product($this->db);
        if ($prod->fetch('', $productRef) <= 0) {
            $this->error = 'Product with ref ' . $productRef . ' not found';
            return -1;
        }

        $label = 'Auto BOM for ' . $productRef . ' (Job ' . (int) $job . ')';

        // Check if already exists
        $sql  = "SELECT rowid FROM " . MAIN_DB_PREFIX . "bom_bom";
        $sql .= " WHERE fk_product = " . ((int) $prod->id);
        $sql .= " AND entity = " . ((int) $conf->entity);
        $sql .= " AND (";
        $sql .= " label = '" . $this->db->escape($label) . "'";
        $sql .= " OR (note_public LIKE '%" . $this->db->escape("Generated by ProductionRules (Downstream)") . "%'";
        $sql .= " AND note_public LIKE '%Job: " . ((int) $job) . "%')";
        $sql .= " )";
        $sql .= " LIMIT 1";
        $resql = $this->db->query($sql);
        if ($resql && ($obj = $this->db->fetch_object($resql))) {
            return 0; // exists, skip
        }

        $bom = new BOM($this->db);
        $bom->label      = $label;
        $bom->quantity   = 1.0;          // producing one unit of target product
        $bom->fk_product = $prod->id;
        $bom->fk_user    = $user->id;
        $bom->entity     = $conf->entity;

        // Note: include job + routing
        $note = array();
        $note[] = 'Generated by ProductionRules (Downstream)';
        $note[] = 'Job: ' . (int) $job;
        $note[] = 'From: ' . $componentProd->ref;
        $note[] = 'Qty SF (for reference): ' . (int) $qty_sf;
        if (!empty($routingLines)) {
            $note[] = '';
            $note[] = 'Routing:';
            foreach ($routingLines as $l) $note[] = $l;
        }
        $bom->note_public = implode("\n", $note);

        $resbom = $bom->create($user);
        if ($resbom <= 0) {
            $this->error = 'Error creating BOM for ' . $productRef . ' : ' . $bom->error;
            return -1;
        }

        // Add component line
        $line = new BOMLine($this->db);
        $line->fk_bom       = $bom->id;
        $line->fk_product   = $componentProd->id;
        $line->qty          = (float) $componentQty;
        $line->fk_unit      = 0;
        $line->position     = 1;
        $line->desc         = 'Auto component from ' . $componentProd->ref;

        $resline = $line->create($user);
        if ($resline <= 0) {
            $this->error = 'Error creating BOM line for component ' . $componentProd->ref . ' : ' . $line->error;
            return -1;
        }

        // Enable
        $bom->status = 1;
        $bom->update($user);

        return 1;
    }

}
