(function () {
  'use strict';

  function $(id) { return document.getElementById(id); }

  function isFullscreen() {
    return !!(document.fullscreenElement ||
              document.webkitFullscreenElement ||
              document.mozFullScreenElement ||
              document.msFullscreenElement);
  }

  function requestFullscreen(el) {
    const fn = el.requestFullscreen ||
               el.webkitRequestFullscreen ||
               el.mozRequestFullScreen ||
               el.msRequestFullscreen;
    if (fn) fn.call(el);
  }

  function exitFullscreen() {
    const fn = document.exitFullscreen ||
               document.webkitExitFullscreen ||
               document.mozCancelFullScreen ||
               document.msExitFullscreen;
    if (fn) fn.call(document);
  }

  function initTooltips() {
    if (!window.jQuery || !jQuery.fn || !jQuery.fn.tooltip) return;

    const $default = jQuery('.classfortooltip').not('.pl-job-tooltip');
    if ($default.length) {
      $default.each(function(){
        try{
          var $el = jQuery(this);
          var inst = $el.tooltip('instance');
          if (inst) { $el.tooltip('destroy'); }
        }catch(e){}
      });
      $default.tooltip();
    }

    const $job = jQuery('.pl-job-tooltip.classfortooltip');
    if ($job.length) {
      $job.each(function(){
        try{
          var $el = jQuery(this);
          var inst = $el.tooltip('instance');
          if (inst) { $el.tooltip('destroy'); }
        }catch(e){}
      });
      $job.tooltip({
        content: function () {
          const t = jQuery(this).attr('title') || '';
          return String(t).replace(/\n|&#10;/g, '<br>');
        },
        position: { my: 'left top+15', at: 'left bottom', collision: 'flipfit' }
      });
    }
  }

  let hideTimer = null;
  function scheduleHideTopbar() {
    clearTimeout(hideTimer);
    hideTimer = setTimeout(function () {
      if (isFullscreen()) document.body.classList.add('pl-top-hidden');
    }, 5000);
  }

  function onFullscreenChange() {
    if (isFullscreen()) scheduleHideTopbar();
    else document.body.classList.remove('pl-top-hidden');
    initTooltips();
  }

  let refreshInterval = 0;
  let countdown = 0;
  let tickTimer = null;

  function startAutoRefresh() {
    const secEl = $('plRefreshSeconds');
    if (!secEl) return;
    refreshInterval = parseInt(secEl.textContent, 10) || 0;
    if (refreshInterval <= 0) return;

    countdown = refreshInterval;
    updateCountdown();

    clearInterval(tickTimer);
    tickTimer = setInterval(function () {
      countdown--;
      if (countdown <= 0) {
        countdown = refreshInterval;
        refreshBoard();
      }
      updateCountdown();
    }, 1000);
  }

  function updateCountdown() {
    const el = $('plRefreshCountdown');
    if (el) el.textContent = String(countdown);
  }

  function refreshBoard() {
    const url = new URL(window.location.href);
    url.searchParams.set('ajax', '1');
    url.searchParams.set('_ts', Date.now());

    fetch(url.toString(), { cache: 'no-store', credentials: 'same-origin' })
      .then(r => r.json())
      .then(data => {
        if (data && data.html) {
          const grid = $('plBoardGrid');
          if (grid) grid.outerHTML = data.html;
        }
        const lu = $('plLastUpdate');
        if (lu) lu.textContent = 'Last update: ' + (data.clock || '-');
        initTooltips();
      })
      .catch(() => {
        const lu = $('plLastUpdate');
        if (lu) lu.textContent = 'Last update: ERROR';
      });
  }

  function init() {
    initTooltips();
    startAutoRefresh();

    const fsBtn = $('plFullscreen');
    if (fsBtn) {
      fsBtn.addEventListener('click', function () {
        const page = $('plBoardPage') || document.documentElement;
        if (!isFullscreen()) requestFullscreen(page);
        else exitFullscreen();
      });
    }

    document.addEventListener('fullscreenchange', onFullscreenChange);
    document.addEventListener('webkitfullscreenchange', onFullscreenChange);
  }

  if (document.readyState === 'loading')
    document.addEventListener('DOMContentLoaded', init);
  else init();
})();
