<?php
require '../../main.inc.php';

require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT.'/mrp/class/mo.class.php';

if (empty($user->rights->planning->read)) accessforbidden();

$langs->loadLangs(array("main", "mrp"));

$pagetitle = "Planning Board";
$stationFilter = trim(GETPOST('station', 'alphanohtml'));
$showDraft = (int) GETPOST('showdraft', 'int');
$autorefresh = (int) GETPOST('refresh', 'int');
if ($autorefresh <= 0) $autorefresh = 60;

$arrayofcss = array();
llxHeader('', $pagetitle, '', '', 0, 0, array(), $arrayofcss);

// --- Workstation map (rowid => ref/label)
$workstationMap = array();       // id => "REF - Label"
$workstationRefMap = array();    // id => "REF"
$wsTables = array(MAIN_DB_PREFIX.'workstation_workstation', MAIN_DB_PREFIX.'workstation', MAIN_DB_PREFIX.'mrp_workstation');
foreach ($wsTables as $wst) {
    // Detect table existence safely
    $sqlchk = "SELECT 1 as ok FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '".$db->escape($wst)."'";
    $reschk = $db->query($sqlchk);
    $ok = ($reschk && ($o = $db->fetch_object($reschk)) && !empty($o->ok));
    if (!$ok) continue;

    $sqlws = "SELECT rowid, ref, label FROM ".$wst;
    $resws = $db->query($sqlws);
    if ($resws) {
        while ($ws = $db->fetch_object($resws)) {
            $id = (int) $ws->rowid;
            $ref = trim((string) $ws->ref);
            $label = trim((string) $ws->label);
            $workstationRefMap[$id] = $ref;
            $display = $ref;
            if ($label !== '') $display .= ' - '.$label;
            $workstationMap[$id] = $display;
        }
        break;
    }
}

// --- Build list of available stations (only live: Validated+In progress (+Draft if toggle))
$availableStations = array(); // raw => label
$sqlst = "SELECT DISTINCT ef.working_station as ws_raw
          FROM ".MAIN_DB_PREFIX."mrp_mo mo
          LEFT JOIN ".MAIN_DB_PREFIX."mrp_mo_extrafields ef ON ef.fk_object = mo.rowid
          WHERE mo.entity IN (".getEntity('mrp_mo').")
            AND ef.working_station IS NOT NULL
            AND ef.working_station <> ''
            AND mo.status IN (1,2".($showDraft ? ",0" : "").")
          ORDER BY ef.working_station ASC";
$resst = $db->query($sqlst);
if ($resst) {
    while ($s = $db->fetch_object($resst)) {
        $raw = (string) $s->ws_raw;
        $lbl = $raw;
        if (ctype_digit($raw)) {
            $id = (int) $raw;
            if (isset($workstationMap[$id])) $lbl = $workstationMap[$id];
        }
        $availableStations[$raw] = $lbl;
    }
    asort($availableStations, SORT_NATURAL | SORT_FLAG_CASE);
}

// --- Main SQL (live view only): statuses Validated + In progress (+ Draft via toggle)
$sql = "SELECT
            mo.rowid,
            mo.ref,
            mo.status,
            mo.qty,
            mo.fk_product,
            mo.date_start_planned,
            mo.date_end_planned,
            (SELECT COALESCE(SUM(CASE WHEN sm.value > 0 THEN sm.value ELSE 0 END),0)
             FROM ".MAIN_DB_PREFIX."stock_mouvement sm
             WHERE sm.origintype IN ('mo','mrp_mo')
               AND sm.fk_origin = mo.rowid
               AND sm.type_mouvement IN (2,3)
            ) as qty_done,
            p.ref as prod_ref,
            p.label as prod_label,
            ef.job_number,
            ef.working_station,
            mo.fk_bom,
            b.note_public as bom_note_public
        FROM ".MAIN_DB_PREFIX."mrp_mo as mo
        LEFT JOIN ".MAIN_DB_PREFIX."mrp_mo_extrafields as ef ON ef.fk_object = mo.rowid
        LEFT JOIN ".MAIN_DB_PREFIX."product as p ON p.rowid = mo.fk_product
        LEFT JOIN ".MAIN_DB_PREFIX."bom_bom as b ON b.rowid = mo.fk_bom
        WHERE mo.entity IN (".getEntity('mrp_mo').")
          AND mo.status IN (1,2".($showDraft ? ",0" : "").")";

if ($stationFilter !== '') {
    $sql .= " AND ef.working_station = '".$db->escape($stationFilter)."' ";
}

$sql .= " ORDER BY ef.working_station ASC, mo.date_start_planned ASC, mo.ref ASC";

$res = $db->query($sql);
if (!$res) {
    print '<div class="error">'.$db->lasterror().'</div>';
    llxFooter();
    exit;
}

// Group by station
$stations = array();
while ($obj = $db->fetch_object($res)) {
    $wsKey = trim((string) $obj->working_station);
    if ($wsKey === '') $wsKey = '(none)';
    if (!isset($stations[$wsKey])) $stations[$wsKey] = array();

    $start = $db->jdate($obj->date_start_planned);
    $end   = $db->jdate($obj->date_end_planned);

    $stations[$wsKey][] = array(
        'rowid' => (int) $obj->rowid,
        'ref' => $obj->ref,
        'status' => (int) $obj->status,
        'qty' => (float) $obj->qty,
        'qty_done' => (float) $obj->qty_done,
        'fk_product' => (int) $obj->fk_product,
        'prod_ref' => $obj->prod_ref,
        'prod_label' => $obj->prod_label,
        'start' => $start,
        'end' => $end,
        'job_number' => $obj->job_number,
        'fk_bom' => (int) $obj->fk_bom,
        'bom_note_public' => $obj->bom_note_public,
    );
}

function ws_ref_label($raw, $workstationRefMap, $workstationMap) {
    $k = trim((string)$raw);
    if ($k === '' || $k === '(none)') return 'UNASSIGNED';
    if (ctype_digit($k)) {
        $id = (int)$k;
        if (isset($workstationRefMap[$id]) && $workstationRefMap[$id] !== '') return $workstationRefMap[$id];
        if (isset($workstationMap[$id]) && $workstationMap[$id] !== '') return $workstationMap[$id];
    }
    return $k;
}

function status_badge($status) {
    // Dolibarr-ish labels
    $map = array(
        0 => array('Draft','pl-status-draft'),
        1 => array('Validated','pl-status-validated'),
        2 => array('In progress','pl-status-inprogress'),
        3 => array('Produced','pl-status-produced'),
        9 => array('Cancelled','pl-status-cancelled'),
    );
    if (!isset($map[$status])) return '<span class="pl-status pl-status-unknown">?</span>';
    return '<span class="pl-status '.$map[$status][1].'">'.$map[$status][0].'</span>';
}

// --- AJAX refresh endpoint (returns JSON with refreshed board grid HTML) ---
$ajax = (int) GETPOST('ajax', 'int');
if ($ajax) {
    ob_start();
    print '<div class="pl-board-grid" id="plBoardGrid">';
    foreach ($stations as $wsKey => $jobs) {
        $wsRef = ws_ref_label($wsKey, $workstationRefMap, $workstationMap);
        $count = count($jobs);

        print '<div class="pl-col">';
        print '<div class="pl-col-head">'.dol_escape_htmltag($wsRef).'<span class="pl-col-count">Jobs ('.$count.')</span></div>';

        foreach ($jobs as $j) {
            $qty = (float)$j['qty'];
            $done = (float)$j['qty_done'];
            $left = max(0, $qty - $done);
            $pct = ($qty > 0) ? min(100, max(0, ($done / $qty) * 100)) : 0;

            $startTxt = $j['start'] ? dol_print_date($j['start'], 'dayhour') : '';
            $endTxt   = $j['end'] ? dol_print_date($j['end'], 'dayhour') : '';
            $duration = '';
            if (!empty($j['start']) && !empty($j['end'])) {
                $seconds = ((int)$j['end']) - ((int)$j['start']);
                if ($seconds > 0) $duration = convertSecondToTime($seconds, 'allhourmin');
            } elseif (!empty($j['start']) && (int)$j['status'] === 2) {
                $seconds = time() - ((int)$j['start']);
                if ($seconds > 0) $duration = convertSecondToTime($seconds, 'allhourmin');
            }

            // MO link
            $moLink = '';
            if (class_exists('Mo')) {
                $moObj = new Mo($db);
                if ($moObj->fetch((int)$j['rowid']) > 0) $moLink = $moObj->getNomUrl(1);
            }
            if (empty($moLink)) {
                $moUrl = DOL_URL_ROOT.'/mrp/mo_card.php?id='.$j['rowid'];
                $moLink = '<a href="'.dol_escape_htmltag($moUrl).'">'.dol_escape_htmltag($j['ref']).'</a>';
            }

            // Product link (standard + label)
            $prodLink = dol_escape_htmltag($j['prod_ref']);
            if (!empty($j['fk_product'])) {
                $pobj = new Product($db);
                if ($pobj->fetch((int)$j['fk_product']) > 0) {
                    $prodLink = $pobj->getNomUrl(1);
                    if (!empty($pobj->label)) $prodLink .= ' - '.dol_escape_htmltag($pobj->label);
                    elseif (!empty($j['prod_label'])) $prodLink .= ' - '.dol_escape_htmltag($j['prod_label']);
                }
            } else {
                if (!empty($j['prod_label'])) $prodLink .= ' - '.dol_escape_htmltag($j['prod_label']);
            }

            // Job link tooltip from BOM note (formatted)
            $jobTxt = (string)$j['job_number'];
            $jobLink = dol_escape_htmltag($jobTxt);
            $bomId = (int)$j['fk_bom'];
            if ($bomId > 0) {
                $noteRaw = (string)$j['bom_note_public'];
                $noteRaw = stripcslashes($noteRaw);
                $noteRaw = str_replace(array('\\r\\n','\\n','\\r'), "\n", $noteRaw);
                $noteRaw = str_replace(array("\r\n","\r"), "\n", $noteRaw);
                $noteRaw = trim($noteRaw);
                $titleHtml = dol_escape_htmltag($noteRaw);
                $titleHtml = str_replace(array('\\r\\n','\\n','\\r'), "<br>", $titleHtml);
                $titleHtml = str_replace(array("\r\n","\n","\r"), "<br>", $titleHtml);
                $urlNote = 'https://develop.for-our.info/bom/bom_note.php?id='.$bomId;
                $jobLink = '<a href="'.dol_escape_htmltag($urlNote).'" class="classfortooltip pl-job-tooltip" data-html="true" title="'.$titleHtml.'">'.dol_escape_htmltag($jobTxt).'</a>';
            }

            print '<div class="pl-card">';
            print '<div class="pl-card-top">';
            print '<div class="pl-card-mo">'.$moLink.'</div>';
            print '<div class="pl-card-status">'.status_badge($j['status']).'</div>';
            print '</div>';

            print '<div class="pl-card-prod">'.$prodLink.'</div>';
            print '<div class="pl-card-job">Job: '.$jobLink.'</div>';

            print '<div class="pl-card-nums">';
            print '<div><span class="pl-k">Qty</span><span class="pl-v">'.dol_escape_htmltag((string)$qty).'</span></div>';
            print '<div><span class="pl-k">Done</span><span class="pl-v">'.dol_escape_htmltag((string)$done).'</span></div>';
            print '<div><span class="pl-k">Left</span><span class="pl-v">'.dol_escape_htmltag((string)$left).'</span></div>';
            print '</div>';

            print '<div class="pl-progress"><div class="pl-progress-bar" style="width:'.((int)$pct).'%"></div></div>';

            print '<div class="pl-card-times">';
            if ($startTxt) print '<span>'.$startTxt.'</span>';
            if ($endTxt) print '<span>→ '.$endTxt.'</span>';
            if ($duration) print '<span class="pl-card-dur">'.$duration.'</span>';
            print '</div>';

            print '</div>'; // card
        }

        print '</div>'; // col
    }
    print '</div>'; // grid
    $gridHtml = ob_get_clean();

    $payload = array(
        'html' => $gridHtml,
        'clock' => dol_print_date(dol_now(), 'dayhoursec'),
    );
    header('Content-Type: application/json; charset=utf-8');
    print json_encode($payload);
    exit;
}

print '<div class="pl-board-page" id="plBoardPage">';

// Top bar
print '<div class="pl-board-top">';
print '<div class="pl-board-title">'.$pagetitle.'</div>';

print '<form class="pl-board-filters" method="GET" action="'.dol_escape_htmltag($_SERVER['PHP_SELF']).'">';
print '<select name="station" class="minwidth200">';
print '<option value="">All stations</option>';
foreach ($availableStations as $raw => $lbl) {
    $sel = ($stationFilter !== '' && (string)$stationFilter === (string)$raw) ? ' selected' : '';
    print '<option value="'.dol_escape_htmltag($raw).'"'.$sel.'>'.dol_escape_htmltag($lbl).'</option>';
}
print '</select>';

$chk = $showDraft ? ' checked' : '';
print '<label class="pl-board-check"><input type="checkbox" name="showdraft" value="1"'.$chk.'> Show Draft</label>';

print '<select name="refresh" class="minwidth100">';
foreach (array(0=>'Off',30=>'30s',60=>'60s',120=>'2m',300=>'5m') as $sec=>$txt) {
    $sel = ((int)$autorefresh === (int)$sec) ? ' selected' : '';
    print '<option value="'.((int)$sec).'"'.$sel.'>'.$txt.'</option>';
}
print '</select>';

print '<button class="butAction" type="submit">Apply</button>';
print '<button class="butAction" type="button" id="plFullscreen">Fullscreen</button>';
print '</form>';

print '<div class="pl-board-meta"><span id="plClock"></span><span class="pl-dot">•</span><span>Auto-refresh: '.((int)$autorefresh).'s (<span id="plRefreshCountdown">'.((int)$autorefresh).'</span>)</span></div>';
print '</div>';

// Board columns
print '<div class="pl-board-grid" id="plBoardGrid">';
foreach ($stations as $wsKey => $jobs) {
    $wsRef = ws_ref_label($wsKey, $workstationRefMap, $workstationMap);
    $count = count($jobs);

    print '<div class="pl-col">';
    print '<div class="pl-col-head">'.dol_escape_htmltag($wsRef).'<span class="pl-col-count">Jobs ('.$count.')</span></div>';

    foreach ($jobs as $j) {
        $qty = (float)$j['qty'];
        $done = (float)$j['qty_done'];
        $left = max(0, $qty - $done);
        $pct = ($qty > 0) ? min(100, max(0, ($done / $qty) * 100)) : 0;

        $startTxt = $j['start'] ? dol_print_date($j['start'], 'dayhour') : '';
        $endTxt   = $j['end'] ? dol_print_date($j['end'], 'dayhour') : '';
        $duration = '';
        if (!empty($j['start']) && !empty($j['end'])) {
            $sec = ((int)$j['end']) - ((int)$j['start']);
            if ($sec > 0) $duration = convertSecondToTime($sec, 'allhourmin');
        } elseif (!empty($j['start']) && (int)$j['status'] === 2) {
            $sec = time() - ((int)$j['start']);
            if ($sec > 0) $duration = convertSecondToTime($sec, 'allhourmin');
        }

        // MO tooltip (standard)
        $moLink = '';
        if (class_exists('Mo')) {
            $moObj = new Mo($db);
            if ($moObj->fetch((int)$j['rowid']) > 0) $moLink = $moObj->getNomUrl(1);
        }
        if (empty($moLink)) {
            $moUrl = DOL_URL_ROOT.'/mrp/mo_card.php?id='.$j['rowid'];
            $moLink = '<a href="'.dol_escape_htmltag($moUrl).'">'.dol_escape_htmltag($j['ref']).'</a>';
        }

        // Product link (standard + label)
        $prodLink = dol_escape_htmltag($j['prod_ref']);
        if (!empty($j['fk_product'])) {
            $pobj = new Product($db);
            if ($pobj->fetch((int)$j['fk_product']) > 0) {
                $prodLink = $pobj->getNomUrl(1);
                if (!empty($pobj->label)) $prodLink .= ' - '.dol_escape_htmltag($pobj->label);
                elseif (!empty($j['prod_label'])) $prodLink .= ' - '.dol_escape_htmltag($j['prod_label']);
            }
        } else {
            if (!empty($j['prod_label'])) $prodLink .= ' - '.dol_escape_htmltag($j['prod_label']);
        }

        // Job link tooltip from BOM note (formatted)
        $jobTxt = (string)$j['job_number'];
        $jobLink = dol_escape_htmltag($jobTxt);
        $bomId = (int)$j['fk_bom'];
        if ($bomId > 0) {
            $noteRaw = (string)$j['bom_note_public'];
            $noteRaw = stripcslashes($noteRaw);
            $noteRaw = str_replace(array('\\r\\n','\\n','\\r'), "\n", $noteRaw);
            $noteRaw = str_replace(array("\r\n","\r"), "\n", $noteRaw);
            $noteRaw = trim($noteRaw);
            $titleHtml = dol_escape_htmltag($noteRaw);
            $titleHtml = str_replace(array('\\r\\n','\\n','\\r'), "<br>", $titleHtml);
            $titleHtml = str_replace(array("\r\n","\n","\r"), "<br>", $titleHtml);
            $urlNote = 'https://develop.for-our.info/bom/bom_note.php?id='.$bomId;
            $jobLink = '<a href="'.dol_escape_htmltag($urlNote).'" class="classfortooltip pl-job-tooltip" data-html="true" title="'.$titleHtml.'">'.dol_escape_htmltag($jobTxt).'</a>';
        }

        print '<div class="pl-card">';
        print '<div class="pl-card-top">';
        print '<div class="pl-card-mo">'.$moLink.'</div>';
        print '<div class="pl-card-status">'.status_badge((int)$j['status']).'</div>';
        print '</div>';

        print '<div class="pl-card-prod">'.$prodLink.'</div>';
        print '<div class="pl-card-job">Job: '.$jobLink.'</div>';

        print '<div class="pl-card-nums">';
        print '<div><span class="pl-k">Qty</span><span class="pl-v">'.dol_escape_htmltag((string)$qty).'</span></div>';
        print '<div><span class="pl-k">Done</span><span class="pl-v">'.dol_escape_htmltag((string)$done).'</span></div>';
        print '<div><span class="pl-k">Left</span><span class="pl-v">'.dol_escape_htmltag((string)$left).'</span></div>';
        print '</div>';

        print '<div class="pl-progress"><div class="pl-progress-bar" style="width:'.((int)$pct).'%"></div></div>';

        print '<div class="pl-card-times">';
        if ($startTxt) print '<span>'.$startTxt.'</span>';
        if ($endTxt) print '<span>→ '.$endTxt.'</span>';
        if ($duration) print '<span class="pl-dur">'.$duration.'</span>';
        print '</div>';

        print '</div>'; // card
    }

    print '</div>'; // col
}
print '</div>'; // grid

print '</div>'; // page

print '<style>
/* Fullscreen board */
.pl-board-page{margin-top:10px;padding:10px;}
.pl-board-top{display:flex;flex-wrap:wrap;align-items:center;gap:10px;margin-bottom:10px;}
.pl-board-title{font-size:20px;font-weight:700;margin-right:auto;}
.pl-board-filters{display:flex;flex-wrap:wrap;gap:10px;align-items:center;}
.pl-board-check{display:flex;align-items:center;gap:6px;font-weight:600;}
.pl-board-meta{margin-left:auto;display:flex;align-items:center;gap:8px;opacity:.8;}
.pl-dot{opacity:.5}

.pl-board-grid{display:grid;grid-auto-flow:column;grid-auto-columns:minmax(280px,1fr);gap:12px;overflow-x:auto;padding-bottom:8px;}
@media (min-width: 1200px){ .pl-board-grid{grid-auto-columns:minmax(320px,1fr);} }
@media (min-width: 1600px){ .pl-board-grid{grid-auto-columns:minmax(360px,1fr);} }

.pl-col{background:rgba(0,0,0,0.02);border-radius:12px;padding:10px;min-height:200px;}
.pl-col-head{display:flex;justify-content:space-between;align-items:baseline;font-weight:800;font-size:16px;margin-bottom:10px;}
.pl-col-count{font-weight:700;opacity:.7;font-size:13px;}

.pl-card{background:#fff;border-radius:12px;padding:10px;box-shadow:0 1px 2px rgba(0,0,0,.08);margin-bottom:10px;}
.pl-card-top{display:flex;justify-content:space-between;align-items:flex-start;gap:8px;}
.pl-card-mo{font-size:15px;font-weight:800;}
.pl-card-prod{margin-top:6px;font-weight:700;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;}
.pl-card-job{margin-top:6px;opacity:.85;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;}
.pl-card-nums{display:flex;gap:12px;margin-top:8px;}
.pl-card-nums .pl-k{display:block;font-size:11px;opacity:.7}
.pl-card-nums .pl-v{display:block;font-size:16px;font-weight:800}

.pl-progress{height:8px;background:rgba(0,0,0,.08);border-radius:999px;overflow:hidden;margin-top:8px;}
.pl-progress-bar{height:100%;background:rgba(25,168,122,.9);}
.pl-card-times{display:flex;flex-wrap:wrap;gap:10px;margin-top:8px;font-size:12px;opacity:.85}
.pl-dur{margin-left:auto;font-weight:800;}

.pl-status{display:inline-block !important;padding:2px 8px !important;border-radius:6px !important;font-weight:600 !important;font-size:0.85em !important;white-space:nowrap !important;}
.pl-status-draft{background:#e5e7eb !important;color:#374151 !important;}
.pl-status-validated{background:#b6921a !important;color:#fff !important;}
.pl-status-inprogress{background:#19a87a !important;color:#fff !important;}
.pl-status-produced{background:#cfd4d7 !important;color:#374151 !important;}
.pl-status-cancelled{background:#f3f4f6 !important;color:#6b7280 !important;}
.pl-status-unknown{background:#9ca3af !important;color:#111827 !important;}

/* Fullscreen tweaks */
body.pl-fullscreen .pl-board-top{position:sticky;top:0;background:rgba(255,255,255,.9);backdrop-filter:blur(6px);padding:8px;border-radius:12px;z-index:10;}
/* Hide Dolibarr chrome in kiosk/fullscreen */
.pl-kiosk #id-left,
.pl-kiosk #id-top,
.pl-kiosk .side-nav,
.pl-kiosk .tmenudiv,
.pl-kiosk #mainmenu,
.pl-kiosk #topmenu,
.pl-kiosk #menu,
.pl-kiosk #id-left-menu,
.pl-fullscreen #id-left,
.pl-fullscreen #id-top,
.pl-fullscreen .side-nav,
.pl-fullscreen .tmenudiv,
.pl-fullscreen #mainmenu,
.pl-fullscreen #topmenu,
.pl-fullscreen #menu,
.pl-fullscreen #id-left-menu{
  display:none !important;
}
.pl-kiosk #id-right,
.pl-fullscreen #id-right{
  margin-left:0 !important;
  left:0 !important;
  width:100% !important;
}
.pl-kiosk #id-container,
.pl-fullscreen #id-container{
  padding-top:0 !important;
  margin-top:0 !important;
}
</style>';

print '<script>
(function(){
  function updateClock(){
    var d=new Date();
    var s=d.toLocaleString();
    var el=document.getElementById("plClock");
    if(el) el.textContent=s;
  }
  updateClock();
  setInterval(updateClock, 1000);

  var fsBtn = document.getElementById("plFullscreen");
var fsDenied = false;

function fsSupport(){
  var el = document.getElementById("plBoardPage") || document.documentElement;
  return !!(el.requestFullscreen || el.webkitRequestFullscreen || el.mozRequestFullScreen || el.msRequestFullscreen);
}
function isFullscreen(){
  return !!(document.fullscreenElement || document.webkitFullscreenElement || document.mozFullScreenElement || document.msFullscreenElement);
}
function setUiModes(){
  var isFs = isFullscreen();
  document.body.classList.toggle("pl-fullscreen", isFs);
  // kiosk stays as-is
}
function updateFsButton(){
  if(!fsBtn) return;
  if(isFullscreen() || document.body.classList.contains("pl-kiosk")) fsBtn.textContent = "Exit";
  else fsBtn.textContent = (fsSupport() && !fsDenied) ? "Fullscreen" : "Kiosk";
}

if(fsBtn){
  updateFsButton();

  fsBtn.addEventListener("click", function(){
    // Prefer real Fullscreen; fallback to kiosk if blocked or unsupported
    if(fsSupport() && !fsDenied){
      var el = document.getElementById("plBoardPage") || document.documentElement;
      if(!isFullscreen()){
        var req = el.requestFullscreen || el.webkitRequestFullscreen || el.mozRequestFullScreen || el.msRequestFullscreen;
        try{
          var ret = req && req.call(el);
          if(ret && typeof ret.then === "function"){
            ret.catch(function(){
              fsDenied = true;
              document.body.classList.add("pl-kiosk");
              updateFsButton();
            });
          }
        }catch(e){
          fsDenied = true;
          document.body.classList.add("pl-kiosk");
        }
      } else {
        var exit = document.exitFullscreen || document.webkitExitFullscreen || document.mozCancelFullScreen || document.msExitFullscreen;
        try{ exit && exit.call(document); }catch(e){}
      }
    } else {
      document.body.classList.toggle("pl-kiosk");
    }
    setTimeout(function(){ setUiModes(); updateFsButton(); }, 200);
  });

  document.addEventListener("fullscreenchange", function(){ setUiModes(); updateFsButton(); });
  document.addEventListener("webkitfullscreenchange", function(){ setUiModes(); updateFsButton(); });
}

var refresh = '.((int)$autorefresh).';
var countdownEl = document.getElementById("plRefreshCountdown");
var refreshTimer = null;
var tickTimer = null;

function doAjaxRefresh(){
  try{
    var url = new URL(window.location.href);
    url.searchParams.set("ajax","1");
    // remove any fragment
    url.hash = "";
    fetch(url.toString(), {cache:"no-store", credentials:"same-origin"})
      .then(function(r){
        if(!r.ok) throw new Error("HTTP "+r.status);
        return r.json();
      })
      .then(function(data){
        if(data && data.html){
          var grid = document.getElementById("plBoardGrid");
          if(grid){
            grid.outerHTML = data.html;
          }
          // Re-init tooltips for new DOM
          if (window.jQuery && jQuery.fn && jQuery.fn.tooltip) {
            jQuery(".classfortooltip").tooltip({ position: { my: "left top+15", at: "left bottom", collision: "flipfit" } });
          }
        }
        var clk = document.getElementById("plClock");
        if(clk && data && data.clock) clk.textContent = data.clock;
      })
      .catch(function(err){
        // show small hint in console for debugging
        try{ console.warn("Planning Board refresh failed", err); }catch(e){}
      });
  }catch(e){
    try{ console.warn("Planning Board refresh exception", e); }catch(ex){}
  }
}

function startCountdown(){
  if(!(refresh > 0)) return;
  var remaining = refresh;
  if(countdownEl) countdownEl.textContent = remaining;
  if(tickTimer) clearInterval(tickTimer);
  tickTimer = setInterval(function(){
    remaining -= 1;
    if(remaining <= 0){
      remaining = refresh;
      doAjaxRefresh();
    }
    if(countdownEl) countdownEl.textContent = remaining;
  }, 1000);
}

if(refresh > 0){
  startCountdown();
}
})();
</script>';

llxFooter();
